package com.distributed.search;

import org.apache.curator.framework.CuratorFramework;
import org.apache.zookeeper.CreateMode;
import java.net.InetAddress;
import java.util.List;

public class ServiceRegistry {
    private static final String WORKERS_REGISTRY_PATH = "/workers";
    private static final String LEADER_REGISTRY_PATH = "/leader";
    private final CuratorFramework client;

    public ServiceRegistry(CuratorFramework client) {
        this.client = client;
    }

    // register worker active
    public void registerWorker(int port) throws Exception {
        if (client.checkExists().forPath(WORKERS_REGISTRY_PATH) == null) {
            client.create().creatingParentsIfNeeded().forPath(WORKERS_REGISTRY_PATH);
        }
        String ipAddress = InetAddress.getLocalHost().getHostAddress();
        String workerPath = WORKERS_REGISTRY_PATH + "/worker_" + ipAddress + ":" + port;

        //node (EPHEMERAL)
        if (client.checkExists().forPath(workerPath) != null) {
            client.delete().forPath(workerPath);
        }
        client.create().withMode(CreateMode.EPHEMERAL).forPath(workerPath);
        System.out.println("[Registry] Worker registered at: " + ipAddress + ":" + port);
    }

    // --- الوظائف الجديدة ---

    /**
     * تسجيل القائد الحالي (يستدعيها الـ Coordinator فقط بعد الفوز بالانتخاب)
     */
    public void registerLeader(int port) throws Exception {
        String ipAddress = InetAddress.getLocalHost().getHostAddress();
        String leaderAddress = ipAddress + ":" + port;

        // إذا كان هناك قائد قديم مسجل، نحذفه (لضمان التحديث)
        if (client.checkExists().forPath(LEADER_REGISTRY_PATH) != null) {
            client.delete().forPath(LEADER_REGISTRY_PATH);
        }

        // إنشاء عقدة القائد كعقدة مؤقتة (EPHEMERAL)
        client.create().withMode(CreateMode.EPHEMERAL).forPath(LEADER_REGISTRY_PATH, leaderAddress.getBytes());
        System.out.println("[Registry] Leader registered globally at: " + leaderAddress);
    }

    /**
     * جلب عنوان القائد (يستخدمها الـ Front-End ليعرف لمن يرسل الـ Query)
     */
    public String getLeaderAddress() throws Exception {
        if (client.checkExists().forPath(LEADER_REGISTRY_PATH) != null) {
            byte[] data = client.getData().forPath(LEADER_REGISTRY_PATH);
            return new String(data);
        }
        return null; // لا يوجد قائد حالياً
    }

    // اكتشاف العمال النشطين (يستخدمه القائد لتوزيع المهام)
    public List<String> getActiveWorkers() throws Exception {
        if (client.checkExists().forPath(WORKERS_REGISTRY_PATH) == null) {
            return List.of();
        }
        return client.getChildren().forPath(WORKERS_REGISTRY_PATH);
    }
}