/*
 * Decompiled with CFR 0.152.
 */
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.ZooKeeper;

public class Autohealer
implements Watcher {
    private static final String ZOOKEEPER_ADDRESS = "192.168.144.134:2181";
    private static final int SESSION_TIMEOUT = 3000;
    private static final String WORKERS_PATH = "/workers";
    private static final String NODES_PATH = "/nodes";
    private static final String LOG_FILE = "cluster_events.log";
    private final int numberOfWorkers;
    private ZooKeeper zooKeeper;
    private int nodeIndex = 0;

    public Autohealer(int numberOfWorkers, String pathToProgram) {
        this.numberOfWorkers = numberOfWorkers;
    }

    public void connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(ZOOKEEPER_ADDRESS, 3000, (Watcher)this);
    }

    public void startWatching() throws KeeperException, InterruptedException {
        this.ensurePathExists(WORKERS_PATH);
        this.ensurePathExists(NODES_PATH);
        this.logEvent("Autohealer started. Monitoring cluster for " + this.numberOfWorkers + " workers.");
        this.checkHealthAndRebalance();
    }

    private void ensurePathExists(String path) throws KeeperException, InterruptedException {
        if (this.zooKeeper.exists(path, false) == null) {
            this.zooKeeper.create(path, new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.PERSISTENT);
        }
    }

    public void process(WatchedEvent event) {
        switch (event.getType()) {
            case None: {
                if (event.getState() != Watcher.Event.KeeperState.SyncConnected) break;
                System.out.println("Successfully connected to Zookeeper");
                break;
            }
            case NodeChildrenChanged: {
                this.logEvent("Change detected in cluster (Nodes or Workers changed).");
                this.checkHealthAndRebalance();
            }
        }
    }

    private void checkHealthAndRebalance() {
        try {
            List currentWorkers = this.zooKeeper.getChildren(WORKERS_PATH, (Watcher)this);
            List availableNodes = this.zooKeeper.getChildren(NODES_PATH, (Watcher)this);
            int currentCount = currentWorkers.size();
            int nodesCount = availableNodes.size();
            System.out.println("Status: Workers = " + currentCount + ", Nodes = " + nodesCount);
            if (currentCount < this.numberOfWorkers) {
                if (nodesCount == 0) {
                    this.logEvent("CRITICAL: All physical nodes are down! Cannot heal cluster.");
                    return;
                }
                this.logEvent("Healing required: " + currentCount + "/" + this.numberOfWorkers + " workers active.");
                int workersToLaunch = this.numberOfWorkers - currentCount;
                this.distributeWorkers(workersToLaunch, availableNodes);
            }
        }
        catch (InterruptedException | KeeperException e) {
            this.logEvent("Error during health check: " + e.getMessage());
            e.printStackTrace();
        }
    }

    private void distributeWorkers(int count, List<String> nodes) throws KeeperException, InterruptedException {
        Collections.sort(nodes);
        for (int i = 0; i < count; ++i) {
            String targetNode = nodes.get(this.nodeIndex % nodes.size());
            ++this.nodeIndex;
            this.sendLaunchCommand(targetNode);
        }
    }

    private void sendLaunchCommand(String nodeName) throws KeeperException, InterruptedException {
        String taskPath = "/nodes/" + nodeName + "/tasks";
        this.ensurePathExists(taskPath);
        String taskNode = this.zooKeeper.create(taskPath + "/task_", new byte[0], (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL);
        this.logEvent("Scheduled new worker on node: " + nodeName + " (Task ID: " + taskNode.substring(taskNode.lastIndexOf("/") + 1) + ")");
    }

    private void logEvent(String message) {
        String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
        String logMessage = "[" + timestamp + "] " + message;
        System.out.println(logMessage);
        try (FileWriter fw = new FileWriter(LOG_FILE, true);
             PrintWriter out = new PrintWriter(fw);){
            out.println(logMessage);
        }
        catch (IOException e) {
            System.err.println("Could not write to log file: " + e.getMessage());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void run() throws InterruptedException {
        ZooKeeper zooKeeper = this.zooKeeper;
        synchronized (zooKeeper) {
            while (this.zooKeeper.getState().isAlive()) {
                this.zooKeeper.wait(5000L);
            }
        }
    }

    public void close() throws InterruptedException {
        this.zooKeeper.close();
    }
}

