import java.util.LinkedList;

public class ThreadPool {
    private final int numThreads;
    private final int queueSize;
    private LinkedList<Runnable> taskQueue;
    private Thread[] threads;
    private boolean isStopped = false;

    public boolean isStopped() {
        return isStopped;
    }

    public ThreadPool(int numThreads, int queueSize) {
        this.numThreads = numThreads;
        this.queueSize = queueSize;
        this.taskQueue = new LinkedList<>();
        this.threads = new Thread[numThreads];

        for (int i = 0; i < numThreads; i++) {
            threads[i] = new Thread(new PoolThread());
            threads[i].start();
        }
    }

    public synchronized void execute(Runnable task) {
        if (isStopped ) {
            throw new IllegalStateException("ThreadPool is stopped");
        }

        // Check if the task queue has reached its limit
        while (taskQueue.size() >= queueSize) {
            try {
                System.out.println("Task queue is full. Waiting for space or timeout...");
                wait();
            } catch (InterruptedException e) {
                // Handle interruption if needed
            }
        }

        taskQueue.addLast(task);
        notify(); // Notify one waiting thread that a new task is available
    }

    public synchronized void waitUntilAllTaskFinished(long timeoutMillis) throws InterruptedException {
        long endTime = System.currentTimeMillis() + timeoutMillis;
        while (System.currentTimeMillis() < endTime) {
            long timeRemaining = endTime - System.currentTimeMillis();
            if (timeRemaining > 0) {
                System.out.println("Waiting for tasks to finish or timeout...");
                wait(timeRemaining);
            } else {
                // Timeout has been reached
                break;
            }
        }
        System.out.println("Timeout has been reached, stop the threads.");
        stop();
    }


    public synchronized void stop() {
        isStopped  = true;
        for (Thread  thread : threads) {
            thread.interrupt(); // Interrupt threads to break out of blocking state
        }
    }

    private class PoolThread implements Runnable {
        @Override
        public void run() {
            while (true) {
                synchronized (ThreadPool.this) {
                    while (taskQueue.isEmpty() && !isStopped ) {
                        try {
                            ThreadPool.this.wait(); // Wait for a task to be added
                        } catch (InterruptedException e) {
                            // Thread interrupted, check if it's time to exit
                            if (isStopped ) {
                                return;
                            }
                        }
                    }

                    if (isStopped) {
                        return;
                    }

                    // Get and execute the next task
                    Runnable task = taskQueue.removeFirst();
                    ThreadPool.this.notify(); // Notify that a slot in the queue is now available
                    task.run();
                }
            }
        }
    }
}
