package handlers.replica;

import command.CommandExecutable;
import command.CommandProcessable;
import factory.replica.ReplicaCommandFactory;
import model.Command;
import util.CommandUtil;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ReplicaCommandHandler implements CommandProcessable {
    private static final Logger logger = Logger.getLogger(ReplicaCommandHandler.class.getName());
    private final List<String> commands;
    private final ReplicaCommandFactory factory = ReplicaCommandFactory.getInstance();

    public ReplicaCommandHandler(List<String> commands) {
        logger.info("replica command handler");
        this.commands = Collections.unmodifiableList(commands);
    }

    @Override
    public void process() {
        if (commands.isEmpty()) {
            logger.warning("No commands to process.");
            return;
        }

        try {
            Command command = extractCommand();
            if (command == null) {
                logger.warning("Invalid command detected.");
                return;
            }
            logger.info("Executing");
            CommandExecutable<?> commandProcessor = factory.getCommand(command, commands.subList(1, commands.size()));
            executeCommand(commandProcessor);
        } catch (Exception e) {
            logger.log(Level.SEVERE, "Failed to process command.", e);
            throw new RuntimeException(e);
        }
    }

    private Command extractCommand() {
        String firstCommand = commands.getFirst();
        return CommandUtil.getCommand(firstCommand);
    }

    private void executeCommand(CommandExecutable<?> commandProcessor) throws IOException {
        if (commandProcessor != null) {
            commandProcessor.execute();
        } else {
            logger.warning("Command processor is null.");
        }
    }

    @Override
    public void run() {
        process();
    }
}
