package handlers;

import command.CommandExecutable;
import command.CommandInvoker;
import command.UnknownCommand;
import factory.CommandFactory;
import model.Command;
import parser.CommandParser;
import server.SendToReplica;
import util.CommandUtil;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class ClientCommandHandler {
    private static final Logger logger = Logger.getLogger(ClientCommandHandler.class.getName());
    private final List<String> commands;
    private final OutputStream os;
    private final CommandParser commandParser;
    private final CommandFactory factory = CommandFactory.getInstance();
    private final SendToReplica sendToReplica;

    public ClientCommandHandler(List<String> commands, OutputStream outputStream) {
        this.commands = Collections.unmodifiableList(commands);
        this.os = outputStream;
        this.commandParser = new CommandParser();
        this.sendToReplica = SendToReplica.getInstance();
        sendToReplica.start();
    }

    public void execute() {
        if (commands.isEmpty()) {
            logger.warning("No commands to execute.");
            return;
        }

        try {
            Command command = CommandUtil.getCommand(commands.getFirst());
            if (command == null) {
                logger.log(Level.SEVERE, "The client send UnRecognized command.");
                CommandExecutable<byte[]> result = new UnknownCommand("UnRecognized command");
                byte[] response = result.execute();
                writeResponse(response);
            }
            List<String> replicaCommand = createReplicaCommand(commands);
            byte[] result = executeCommand(command);
            sendReplicaCommand(replicaCommand);
            writeResponse(result);
        } catch (IOException e) {
            logger.log(Level.SEVERE, "Command execution failed", e);
            throw new RuntimeException(e);
        }
    }

    private byte[] executeCommand(Command command) throws IOException {
        CommandExecutable<byte[]> commandProcessor = factory.getCommand(command, commands.subList(1, commands.size()), sendToReplica, os);
        return CommandInvoker.invoke(commandProcessor);
    }

    private void sendReplicaCommand(List<String> replicaCommand) {
        sendToReplica.addCommand(commandParser.getResponseFromCommandArray(replicaCommand));
    }

    private void writeResponse(byte[] result) throws IOException {
        os.write(result);
        os.flush();
    }

    private List<String> createReplicaCommand(List<String> commands) {
        return List.copyOf(commands);
    }
}
