package server;

import client.Client;
import client.master.MasterClient;
import storage.StorageManager;

import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class Server implements AutoCloseable {
    private static final Logger logger = Logger.getLogger(Server.class.getName());
    private final int PORT;
    private final String role;
    private final ExecutorService executor;
    private final StorageManager manager;
    private final ReplicaConnectionService replicaConnectionService;

    Server(int port, String role, ReplicaConnectionService replicaConnectionService) {
        logger.info("Creating Server instance");
        this.PORT = port;
        this.role = role;
        this.executor = Executors.newVirtualThreadPerTaskExecutor();
        this.manager = new StorageManager();
        this.replicaConnectionService = replicaConnectionService;
    }

    public void start() {
        logger.info("Starting server on port " + PORT + " with role " + role);
        if ("slave".equals(role) && replicaConnectionService != null) {
            replicaConnectionService.checkConnection();
        }

        try (ServerSocket serverSocket = new ServerSocket(PORT)) {
            serverSocket.setReuseAddress(true);
            logger.info("Server started on Port " + PORT);

            while (!executor.isShutdown()) {
                try {
                    Socket clientSocket = serverSocket.accept();
                    logger.info("New client connected: " + clientSocket.getRemoteSocketAddress());
                    handleClient(clientSocket);
                } catch (IOException e) {
                    logger.log(Level.SEVERE, "Failed to accept client connection", e);
                }
            }
        } catch (IOException e) {
            logger.log(Level.SEVERE, "Server failed to start", e);
        }
    }

    private void handleClient(Socket clientSocket) {
        executor.submit(() -> {
            try {
                Client task = new MasterClient(clientSocket);
                task.run();
            } catch (Exception e) {
                logger.log(Level.SEVERE, "Error handling client", e);
            } finally {
                try {
                    clientSocket.close();
                } catch (IOException e) {
                    logger.log(Level.WARNING, "Failed to close client socket", e);
                }
            }
        });
    }

    @Override
    public void close() {
        logger.info("Shutting down server");
        if (!executor.isShutdown()) {
            executor.shutdown();
        }
        manager.shutdown();
    }
}
