package storage;

import java.util.Map;
import java.util.Queue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;

public class LRUCachePolicy<K, V> implements CachePolicy<K, V> {
    private final int threshold;
    private final Map<K, V> cacheMap;
    private final Queue<K> accessOrder;

    public LRUCachePolicy(int maxCapacity, int threshold) {
        this.cacheMap = new ConcurrentHashMap<>(maxCapacity);
        this.threshold = threshold;
        this.accessOrder = new ConcurrentLinkedQueue<>();
    }

    @Override
    public void add(K key, V value) {
        cacheMap.put(key, value);
        updateAccessOrder(key);
    }

    @Override
    public V retrieve(K key) {
        V value = cacheMap.get(key);
        if (value != null) {
            updateAccessOrder(key);
        }
        return value;
    }

    @Override
    public void delete(K key) {
        cacheMap.remove(key);
        accessOrder.remove(key);
    }

    @Override
    public void runMaintenance() {
        while (cacheMap.entrySet().size() > threshold) {
            evictLeastRecentlyUsed();
        }
    }

    private void updateAccessOrder(K key) {
        accessOrder.remove(key);
        accessOrder.offer(key);
    }

    private void evictLeastRecentlyUsed() {
        while (!accessOrder.isEmpty()) {
            K leastRecentKey = accessOrder.poll();
            if (cacheMap.containsKey(leastRecentKey)) {
                delete(leastRecentKey);
                break;
            }
        }
    }
}