package server;

import model.Command;
import model.ConnectedReplica;

import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class SendToReplica implements Closeable {
    private static final Logger logger = Logger.getLogger(SendToReplica.class.getName());
    private final Queue<ConnectedReplica> connectedReplicas;
    private final Queue<String> commands;
    private final ExecutorService executorService;
    private final AtomicInteger countCommands;

    private SendToReplica() {
        this.connectedReplicas = new ConcurrentLinkedQueue<>();
        this.commands = new ConcurrentLinkedQueue<>();
        this.executorService = Executors.newSingleThreadExecutor();
        this.countCommands = new AtomicInteger(0);
    }

    public static SendToReplica getInstance() {
        return SendToReplicaHolder.INSTANCE;
    }


    public void start() {
        executorService.submit(this::processCommands);
    }

    private void processCommands() {
        while (true) {

                String command = commands.poll();
                if (command == null) {
                    continue;
                }

                byte[] commandBytes = command.getBytes();
                connectedReplicas.forEach(replica -> {
                    try {
                        OutputStream outputStream = replica.os();
                        outputStream.write(commandBytes);
                        outputStream.flush();
                    } catch (IOException e) {
                        logger.log(Level.SEVERE, "Failed to send command to replica ", e);
                        connectedReplicas.remove(replica);
                    }
                });
        }
    }

    public void addCommand(String command) {
        String lowerCaseCommand = command.toLowerCase();
        if (lowerCaseCommand.contains(Command.SET.getValue().toLowerCase()) || lowerCaseCommand.contains("getack")) {
            countCommands.incrementAndGet();
            commands.add(command);
        }
    }


    public void addConnection(OutputStream outputStream) {
        connectedReplicas.add(new ConnectedReplica(outputStream));
        logger.info("SIZE CONNECTED REPLICAS = " + connectedReplicas.size());
    }

    @Override
    public void close() {
        executorService.shutdown();
        logger.info("Closing replicas");
    }

    private static final class SendToReplicaHolder {
        private static final SendToReplica INSTANCE = new SendToReplica();
    }
}
