package server;

import client.Client;
import client.replica.ReplicaClient;
import handlers.replica.ConnectionHandler;
import parser.CommandParser;

import java.io.BufferedReader;
import java.io.IOException;
import java.net.Socket;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class ReplicaConnectionService implements AutoCloseable {
    private static final Logger logger = Logger.getLogger(ReplicaConnectionService.class.getName());
    private final CommandParser commandParser;
    private final int port;
    private Socket socket;
    private final ExecutorService executorService;

    public ReplicaConnectionService(String[] masterPortAndHost, int port) throws IOException {
        this.port = port;
        this.commandParser = new CommandParser();
        this.executorService = Executors.newSingleThreadExecutor();

        if (masterPortAndHost.length > 0) {
            this.socket = new Socket(masterPortAndHost[0], Integer.parseInt(masterPortAndHost[1]));
        }
    }

    public void getConnection() {
        if (socket == null) {
            logger.warning("Socket is not initialized.");
            return;
        }

        ConnectionHandler connectionHandler = new ConnectionHandler(socket, commandParser, port);
        try (BufferedReader bufferedReader = connectionHandler.handleConnection()) {
            logger.info("Ending handshake");
            Client client = new ReplicaClient(bufferedReader, socket);
            executorService.submit(client::run);
            logger.info("ReplicaConnectionService: CONNECTION SUCCESS");
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

    }

    public void checkConnection() {
        if (socket != null && !socket.isClosed()) {
            getConnection();
        } else {
            logger.warning("Socket is either null or closed.");
        }
    }

    @Override
    public void close() {
        executorService.shutdown();
        if (socket != null && !socket.isClosed()) {
            try {
                socket.close();
            } catch (IOException e) {
                logger.log(Level.SEVERE, "Failed to close socket", e);
            }
        }
    }
}
