package storage;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.StampedLock;

public class LRUCachePolicy<K, V> implements CachePolicy<K, V> {
    private final int maxCapacity;
    private final int threshold;
    private final Map<K, Node<K, V>> cacheMap;
    private final StampedLock lock;
    private final AtomicInteger size;
    private final AtomicReference<Node<K, V>> head;
    private final AtomicReference<Node<K, V>> tail;

    public LRUCachePolicy(int maxCapacity, int threshold) {
        this.maxCapacity = maxCapacity;
        this.threshold = threshold;
        this.cacheMap = new ConcurrentHashMap<>(maxCapacity);
        this.lock = new StampedLock();
        this.size = new AtomicInteger(0);
        this.head = new AtomicReference<>(null);
        this.tail = new AtomicReference<>(null);
        RdbFileReader<String, String> reader = new RdbFileReader<>();
        Map<? extends K, ? extends V> initialData = (Map<? extends K, ? extends V>) reader.readFile();
        for (Map.Entry<? extends K, ? extends V> entry : initialData.entrySet()) {
            add(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public void add(K key, V value) {
        Node<K, V> newNode = new Node<>(key, value);
        Node<K, V> oldNode = cacheMap.put(key, newNode);

        if (oldNode == null) {
            if (size.incrementAndGet() > maxCapacity) {
                evictLeastRecentlyUsed();
            }
            addToFrontOptimistic(newNode);
        } else {
            oldNode.setValue(value);
            moveToFrontOptimistic(oldNode);
        }
    }

    @Override
    public V retrieve(K key) {
        Node<K, V> node = cacheMap.get(key);
        if (node != null) {
            moveToFrontOptimistic(node);
            return node.value;
        }
        return null;
    }

    @Override
    public void delete(K key) {
        Node<K, V> node = cacheMap.remove(key);
        if (node != null) {
            removeNodeOptimistic(node);
            size.decrementAndGet();
        }
    }

    @Override
    public void runMaintenance() {
        while (size.get() > threshold) {
            evictLeastRecentlyUsed();
        }
    }

    private void addToFrontOptimistic(Node<K, V> node) {
        long stamp = lock.tryOptimisticRead();
        Node<K, V> oldHead = head.get();
        node.next = oldHead;

        if (!lock.validate(stamp)) {
            stamp = lock.writeLock();
            try {
                oldHead = head.get();
                node.next = oldHead;
                if (oldHead != null) {
                    oldHead.prev = node;
                } else {
                    tail.set(node);
                }
                head.set(node);
            } finally {
                lock.unlockWrite(stamp);
            }
        } else {
            if (oldHead != null) {
                oldHead.prev = node;
            } else {
                tail.set(node);
            }
            head.set(node);
        }
    }

    private void moveToFrontOptimistic(Node<K, V> node) {
        long stamp = lock.tryOptimisticRead();
        if (node != head.get()) {
            Node<K, V> oldHead = head.get();
            Node<K, V> oldTail = tail.get();

            if (!lock.validate(stamp)) {
                stamp = lock.writeLock();
                try {
                    if (node != head.get()) {
                        if (node.prev != null) {
                            node.prev.next = node.next;
                        }
                        if (node.next != null) {
                            node.next.prev = node.prev;
                        } else {
                            tail.set(node.prev);
                        }
                        oldHead = head.get();
                        node.next = oldHead;
                        node.prev = null;
                        if (oldHead != null) {
                            oldHead.prev = node;
                        }
                        head.set(node);
                        if (tail.get() == null) {
                            tail.set(node);
                        }
                    }
                } finally {
                    lock.unlockWrite(stamp);
                }
            } else {
                if (node.prev != null) {
                    node.prev.next = node.next;
                }
                if (node.next != null) {
                    node.next.prev = node.prev;
                } else if (node == oldTail) {
                    tail.compareAndSet(node, node.prev);
                }

                node.next = oldHead;
                node.prev = null;
                if (oldHead != null) {
                    oldHead.prev = node;
                }
                head.compareAndSet(oldHead, node);
            }
        }
    }

    private void removeNodeOptimistic(Node<K, V> node) {
        long stamp = lock.tryOptimisticRead();
        Node<K, V> prevNode = node.prev;
        Node<K, V> nextNode = node.next;

        if (!lock.validate(stamp)) {
            stamp = lock.writeLock();
            try {
                if (node.prev != null) {
                    node.prev.next = node.next;
                } else {
                    head.set(node.next);
                }
                if (node.next != null) {
                    node.next.prev = node.prev;
                } else {
                    tail.set(node.prev);
                }
            } finally {
                lock.unlockWrite(stamp);
            }
        } else {
            if (prevNode != null) {
                prevNode.next = nextNode;
            } else {
                head.compareAndSet(node, nextNode);
            }
            if (nextNode != null) {
                nextNode.prev = prevNode;
            } else {
                tail.compareAndSet(node, prevNode);
            }
        }
    }

    private void evictLeastRecentlyUsed() {
        Node<K, V> nodeToEvict;
        long stamp = lock.writeLock();
        try {
            nodeToEvict = tail.get();
            if (nodeToEvict != null) {
                Node<K, V> newTail = nodeToEvict.prev;
                tail.set(newTail);
                if (newTail != null) {
                    newTail.next = null;
                } else {
                    head.set(null);
                }
            }
        } finally {
            lock.unlockWrite(stamp);
        }

        if (nodeToEvict != null) {
            cacheMap.remove(nodeToEvict.key);
            size.decrementAndGet();
        }
    }

    private static class Node<K, V> {
        final K key;
        volatile V value;
        volatile Node<K, V> prev;
        volatile Node<K, V> next;

        Node(K key, V value) {
            this.key = key;
            this.value = value;
        }

        void setValue(V value) {
            this.value = value;
        }
    }
}
