package storage;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class Storage implements Cacheable<String, String> {
    private final int capacity = 1000000;
    private final CachePolicy<String, String> storage;
    private final Map<String, Long> timeToExpiration = new ConcurrentHashMap<>(capacity);
    private final Map<String, Long> currentTimeForKey = new ConcurrentHashMap<>(capacity);

    private Storage() {
        this.storage =CachePolicyFactory.getPolicy(capacity);
        RdbFileReader<String, String> reader = new RdbFileReader<>();
        Map<String, Long> stringStringMap = reader.getKeysExpiration();
        this.timeToExpiration.putAll(stringStringMap);
    }

    private static final class StorageHolder {
        private static final Storage instance = new Storage();
    }

    public static Cacheable<String,String> getInstance() {
        return StorageHolder.instance;
    }

    @Override
    public void save(String key, String value) {
        storage.add(key, value);
    }

    @Override
    public void save(String key, String value, Long expirationTime) {
        currentTimeForKey.put(key, System.currentTimeMillis());
        timeToExpiration.put(key, expirationTime);
        save(key, value);
    }

    @Override
    public String get(String key) {
        if (isExpired(key)) {
            delete(key);
            return "";
        }
        return storage.retrieve(key);
    }

    private boolean isExpired(String key) {
        if (currentTimeForKey.containsKey(key)) {
            long currTime = System.currentTimeMillis();
            long expirationTime = timeToExpiration.get(key);
            long currExpiration = currentTimeForKey.get(key);
            return currTime - currExpiration > expirationTime;
        }
        return false;
    }

    @Override
    public void delete(String key) {
        storage.delete(key);
        timeToExpiration.remove(key);
        currentTimeForKey.remove(key);
    }
    @Override
   public  void runCachePolicy() {
        storage.runMaintenance();
    }
}