package storage;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

public class Storage {
    private final int capacity = 1000000;
    private static final Logger logger = Logger.getLogger(Storage.class.getName());
    private final CachePolicy<String, String> storage;
    private final Map<String, Long> timeToExpiration = new ConcurrentHashMap<>(capacity);
    private final Map<String, Long> currentTimeForKey = new ConcurrentHashMap<>(capacity);

    private Storage(int threshold) {
        this.storage = new LRUCachePolicy<>(capacity, threshold);
        RdbFileReader<String, String> reader = new RdbFileReader<>();
        Map<String, Long> stringStringMap = reader.getKeysExpiration();
        this.timeToExpiration.putAll(stringStringMap);
    }

    private static final class StorageHolder {
        private static final Storage instance = new Storage(800000);
    }

    public static Storage getInstance() {
        return StorageHolder.instance;
    }

    public void save(String key, String value) {
        storage.add(key, value);
    }

    public void save(String key, String value, Long expirationTime) {
        currentTimeForKey.put(key, System.currentTimeMillis());
        timeToExpiration.put(key, expirationTime);
        save(key, value);
    }

    public String get(String key) {
        if (isExpired(key)) {
            remove(key);
            return "";
        }
        return storage.retrieve(key);
    }

    private boolean isExpired(String key) {
        if (timeToExpiration.containsKey(key) && !currentTimeForKey.containsKey(key)) {
            long currentTime = System.currentTimeMillis();
            long expirationTime = timeToExpiration.get(key);
            if (currentTime > expirationTime) {
                logger.info("Key expired: " + key);
                return true;
            }
        }
        if (currentTimeForKey.containsKey(key)) {
            long currTime = System.currentTimeMillis();
            long expirationTime = timeToExpiration.get(key);
            long currExpiration = currentTimeForKey.get(key);
            return currTime - currExpiration > expirationTime;
        }
        return false;
    }

    private void remove(String key) {
        storage.delete(key);
        timeToExpiration.remove(key);
        currentTimeForKey.remove(key);
    }

    void runCachePolicy() {
        storage.runMaintenance();
    }
}