package command.handshake;

import model.Command;
import parser.CommandParser;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.OutputStream;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ReplConfCommandSender extends CommandSender {
    private static final Logger logger = Logger.getLogger(ReplConfCommandSender.class.getName());
    private final String port;

    public ReplConfCommandSender(String port, CommandParser commandParser) {
        super(commandParser);
        this.port = port;
    }

    @Override
    public void sendCommand(BufferedReader bufferedReader, OutputStream outputStream) throws IOException {
        try {
            logger.info("Starting REPLCONF command sequence.");

            if (sendReplConfResp(bufferedReader, outputStream)) {
                logger.info("REPLCONF listening-port sent successfully.");

                if (sendReplConfCapa(bufferedReader, outputStream)) {
                    logger.info("REPLCONF capa sent successfully.");

                    sendPsyncCommand(outputStream);
                    logger.info("PSYNC command sent successfully.");
                }
            }

        } catch (IOException e) {
            logger.log(Level.SEVERE, "Failed to send REPLCONF command sequence: " + e.getMessage(), e);
            throw new IOException("Failed to send REPLCONF command: " + e.getMessage(), e);
        }
    }

    private boolean sendReplConfCapa(BufferedReader bufferedReader, OutputStream outputStream) throws IOException {
        try {
            outputStream.write(commandParser.getResponseFromCommandArray(List.of(
                            Command.REPLCONF.getValue(),
                            "capa",
                            "npsync2")
                    ).getBytes()
            );
            outputStream.flush();

            String response = bufferedReader.readLine();
            if (response == null) {
                throw new IOException("No response received after sending REPLCONF capa.");
            }

            logger.info("Received response after REPLCONF capa: " + response);
            return !response.isEmpty();
        } catch (IOException e) {
            logger.log(Level.WARNING, "Error during sending REPLCONF capa: " + e.getMessage(), e);
            throw e;
        }
    }

    private boolean sendReplConfResp(BufferedReader bufferedReader, OutputStream outputStream) throws IOException {
        try {
            outputStream.write(commandParser.getResponseFromCommandArray(List.of(
                            Command.REPLCONF.getValue(),
                            "listening-port",
                            String.valueOf(port))
                    ).getBytes()
            );
            outputStream.flush();

            String response = bufferedReader.readLine();
            if (response == null) {
                throw new IOException("No response received after sending REPLCONF listening-port.");
            }

            logger.info("Received response after REPLCONF listening-port: " + response);
            return !response.isEmpty();
        } catch (IOException e) {
            logger.log(Level.WARNING, "Error during sending REPLCONF listening-port: " + e.getMessage(), e);
            throw e;
        }
    }

    private void sendPsyncCommand(OutputStream outputStream) throws IOException {
        try {
            outputStream.write(commandParser.getResponseFromCommandArray(List.of(
                            Command.PSYNC.getValue(),
                            "?",
                            "-1"))
                    .getBytes()
            );
            outputStream.flush();
        } catch (IOException e) {
            logger.log(Level.WARNING, "Error during sending PSYNC command: " + e.getMessage(), e);
            throw e;
        }
    }
}
