import * as d3 from 'd3';
import { Chart } from "./chart";
import { ChartConfiguration } from "./chartConfiguration";

export class AreaChart extends Chart {
    xScale: any;
    yScale: any;
    xAxis: any;
    yAxis: any;
    xAxisGroup: any;
    yAxisGroup: any;
    lineGenerator: d3.Line<[number, number]>;
    areaGenerator: d3.Area<[number, number]>;
    xValue: (d: any) => any;
    yValue: (d: any) => any;


    constructor(_config: ChartConfiguration, _data?: any[]) {
        super(_config, _data);
        this.initVis();
    }
    protected getDefaultMargins(): { top: number; right: number; bottom: number; left: number; } {
        return {
            top: 20,
            right: 30,
            bottom: 30,
            left: 50
        };

    }
    protected getDefaultContainerSize(): { width: number; height: number; } {
        return {
            width: 800,
            height: 500
        };

    }
    protected initVis(): void {
        const vis = this;

        // Initialize scales (output range only)
        // Input domain will be updated in updateVis method
        // Set up scales
        vis.xScale = d3.scaleTime()
            .range([0, vis.width]);

        vis.yScale = d3.scaleLinear()
            .nice()
            // Reverse for ordering
            .range([vis.height, 0]);

        // Initialize axes
        vis.xAxis = d3.axisBottom(vis.xScale);
        vis.yAxis = d3.axisLeft(vis.yScale);

        // Append empty x-axis group and move it to the bottom of the chart
        vis.xAxisGroup = vis.chart.append("g")
            .attr("transform", "translate(0," + vis.height + ")")
            .call(vis.xAxis);

        // Append y-axis group
        vis.yAxisGroup = vis.chart.append("g")
            .call(vis.yAxis);

        // Define line generator
        vis.lineGenerator = d3.line()
            .x((d: any) => vis.xScale(d.date))
            .y((d: any) => vis.yScale(d.close));

        // Define area generator
        vis.areaGenerator = d3.area()
            .x((d: any) => vis.xScale(d.date))
            .y0(vis.height)
            .y1((d: any) => vis.yScale(d.close));

    }
    protected renderVis(): void {
        const vis = this;

        // Bind data to area paths
        const areaSelection = vis.chart.selectAll(".area")
            .data([vis.data]); // Wrap data in an array for a single area

        // Enter selection for area
        areaSelection.enter()
            .append("path")
            .attr("class", "area")
            .attr("d", vis.areaGenerator)
            .transition()
            .duration(1000)
            .attr("opacity", 1);

        // Update selection for area
        areaSelection
            .transition()
            .duration(1000)
            .attr("d", vis.areaGenerator);

        // Exit selection for area
        areaSelection.exit()
            .transition()
            .duration(1000)
            .attr("opacity", 0)
            .remove();

        // Bind data to line paths
        const lineSelection = vis.chart.selectAll(".line")
            .data([vis.data]);

        // Enter selection for line
        lineSelection.enter()
            .append("path")
            .attr("class", "line")
            .attr("d", vis.lineGenerator)
            .transition()
            .duration(1000)
            .attr("opacity", 1);

        // Update selection for line
        lineSelection
            .transition()
            .duration(1000)
            .attr("d", vis.lineGenerator);

        // Exit selection for line
        lineSelection.exit()
            .transition()
            .duration(1000)
            .attr("opacity", 0)
            .remove();

        // Update the axes/gridlines
        vis.xAxisGroup
            .transition()
            .duration(1000)
            .call(vis.xAxis);

        vis.yAxisGroup
            .transition()
            .duration(1000)
            .call(vis.yAxis);

    }
    public updateVis(): void {
        const vis = this;

        // Specificy accessor functions
        vis.xValue = (d: any) => d.date;
        vis.yValue = (d: any) => d.close;

        // Set the scale input domains
        vis.xScale.domain(d3.extent(vis.data, vis.xValue));
        vis.yScale.domain(d3.extent(vis.data, vis.yValue));

        vis.renderVis();

    }

}