import * as d3 from 'd3';
import { Chart } from "./chart";
import { ChartConfiguration } from "./chartConfiguration";

export class AreaChart extends Chart {
    private xScale: any;
    private yScale: any;
    private xAxis: any;
    private yAxis: any;
    private xAxisGroup: any;
    private yAxisGroup: any;
    private lineGenerator: d3.Line<[number, number]>;
    private xValue: (d: any) => any;
    private yValue: (d: any) => any;
    private tooltip: d3.Selection<d3.BaseType, unknown, HTMLElement, any>;
    private bisectDate: (array: ArrayLike<any>, x: any, lo?: number, hi?: number) => number;
    private tooltipCircle: any;


    constructor(_config: ChartConfiguration, _data?: any[]) {
        super(_config, _data);
        this.bisectDate = d3.bisector((d: any) => d.date).left;
        this.initVis();
    }
    protected getDefaultMargins(): { top: number; right: number; bottom: number; left: number; } {
        return {
            top: 20,
            right: 30,
            bottom: 30,
            left: 50
        };

    }
    protected getDefaultContainerSize(): { width: number; height: number; } {
        return {
            width: 800,
            height: 500
        };

    }
    protected initVis(): void {
        const vis = this;

        // Initialize scales (output range only)
        // Input domain will be updated in updateVis method
        // Set up scales
        vis.xScale = d3.scaleTime()
            .range([0, vis.width]);

        vis.yScale = d3.scaleLinear()
            .nice()
            // Reverse for ordering
            .range([vis.height, 0]);

        // Initialize axes
        vis.xAxis = d3.axisBottom(vis.xScale);
        vis.yAxis = d3.axisLeft(vis.yScale);

        // Append empty x-axis group and move it to the bottom of the chart
        vis.xAxisGroup = vis.chart.append("g")
            .attr("transform", "translate(0," + vis.height + ")")
            .call(vis.xAxis);

        // Append y-axis group
        vis.yAxisGroup = vis.chart.append("g")
            .call(vis.yAxis);

        // Define line generator
        vis.lineGenerator = d3.line()
            .x((d: any) => vis.xScale(d.date))
            .y((d: any) => vis.yScale(d.close));

        // tooltip
        vis.tooltip = d3.select("#line-chart-tooltip");
        vis.tooltipCircle = vis.chart.append("circle")
            .attr("r", 5)
            .attr("stroke", "green")
            .attr("fill", "white")
            .style("display", "none");
    }
    protected renderVis(): void {
        const vis = this;
        // Bind data to line paths
        const lineSelection = vis.chart.selectAll(".line")
            .data([vis.data]);

        // Enter selection for line
        lineSelection.enter()
            .append("path")
            .attr("class", "line")
            .attr("d", vis.lineGenerator)
            .transition()
            .duration(1000)
            .attr("opacity", 1);

        // Update selection for line
        lineSelection
            .transition()
            .duration(1000)
            .attr("d", vis.lineGenerator);

        // Exit selection for line
        lineSelection.exit()
            .transition()
            .duration(1000)
            .attr("opacity", 0)
            .remove();

        // Update the axes/gridlines
        vis.xAxisGroup
            .transition()
            .duration(1000)
            .call(vis.xAxis);

        vis.yAxisGroup
            .transition()
            .duration(1000)
            .call(vis.yAxis);

        // Create a transparent tracking area
        vis.chart.append('rect')
            .attr('width', vis.width)
            .attr('height', vis.height)
            .attr('fill', 'none')
            .attr('pointer-events', 'all')
            .on('mouseenter', () => {
                vis.tooltip.style('display', 'block');
                vis.tooltipCircle.style('display', 'block');
            })
            .on('mouseleave', () => {
                vis.tooltip.style('display', 'none');
                vis.tooltipCircle.style('display', 'none');
            })
            .on('mousemove', (event: any) => {
                const xPos = d3.pointer(event, this)[0] - vis.config.margin.left;

                const date = vis.xScale.invert(xPos);

                // Use the bisector to find the nearest left point
                const index = vis.bisectDate(vis.data, date, 1);
                const a = vis.data[index - 1];
                const b = vis.data[index];
                const d = b && (date - a.date > b.date - date) ? b : a; // seelect nearest point

                if (d) {
                    vis.tooltip.transition()
                    vis.tooltip.html(`Close: ${d.close}`)
                        .style("left", (event.pageX + 5) + "px")
                        .style("top", (event.pageY - 28) + "px");
                    vis.tooltipCircle
                        .attr("cx", vis.xScale(d.date))
                        .attr("cy", vis.yScale(d.close));
                }
            });
    }
    public updateVis(): void {
        const vis = this;

        // Specificy accessor functions
        vis.xValue = (d: any) => d.date;
        vis.yValue = (d: any) => d.close;

        // Set the scale input domains
        vis.xScale.domain(d3.extent(vis.data, vis.xValue));
        vis.yScale.domain(d3.extent(vis.data, vis.yValue));

        vis.renderVis();

    }

}