import * as d3 from 'd3';
import { Chart } from "./chart";
import { ChartConfiguration } from './chartConfiguration';

export class Scatterplot extends Chart {

    private colorScale: d3.ScaleOrdinal<string, unknown, never>;
    private xScale: d3.ScaleLinear<number, number, never>;
    private yScale: d3.ScaleLinear<number, number, never>;
    private xAxis: d3.Axis<d3.NumberValue>;
    private yAxis: d3.Axis<d3.NumberValue>;
    private xAxisG: any;
    private yAxisG: any;
    private colorValue: (d: any) => any;
    private xValue: (d: any) => any;
    private yValue: (d: any) => any;
    private idValue: (d: any) => any;
    private tooltip: d3.Selection<d3.BaseType, unknown, HTMLElement, any>;

    constructor(_config: ChartConfiguration, _data?: any[]) {
        super(_config, _data);
        this.initVis();
    }

    protected getDefaultMargins(): { top: number; right: number; bottom: number; left: number; } {
        return { top: 25, right: 20, bottom: 20, left: 35 };
    }
    protected getDefaultContainerSize(): { width: number; height: number; } {
        return { width: 600, height: 400 };
    }

    protected initVis(): void {
        const vis = this;

        vis.colorScale = d3.scaleOrdinal()
            .range(['#d3eecd', '#7bc77e', '#2a8d46']) // light green to dark green
            .domain(['Easy', 'Intermediate', 'Difficult']);
        vis.xScale = d3.scaleLinear()
            .range([0, vis.width]);
        vis.yScale = d3.scaleLinear()
            .range([vis.height, 0]);

        // Initialize axes
        vis.xAxis = d3.axisBottom(vis.xScale)
            .ticks(6)
            .tickSize(-vis.height - 10)
            .tickPadding(10)
            .tickFormat(d => d + ' km');


        vis.yAxis = d3.axisLeft(vis.yScale)
            .ticks(6)
            .tickSize(-vis.width - 10)
            .tickPadding(10);

        // Append empty x-axis group and move it to the bottom of the chart
        vis.xAxisG = vis.chart.append('g')
            .attr('class', 'axis x-axis')
            .attr('transform', `translate(0,${vis.height})`);

        // Append y-axis group
        vis.yAxisG = vis.chart.append('g')
            .attr('class', 'axis y-axis');

        // Append both axis titles
        vis.chart.append('text')
            .attr('class', 'axis-title')
            .attr('y', vis.height - 15)
            .attr('x', vis.width + 10)
            .attr('dy', '.71em')
            .style('text-anchor', 'end')
            .text('Distance');

        vis.svg.append('text')
            .attr('class', 'axis-title')
            .attr('x', 0)
            .attr('y', 0)
            .attr('dy', '.71em')
            .text('Hours');

        vis.tooltip = d3.select("#scatterplot-tooltip");
    }

    public updateVis(): void {
        let vis = this;
        vis.data.forEach((d: any) => {
            d.time = +d.time;
            d.distance = +d.distance;
        });
        // Specificy accessor functions
        vis.colorValue = d => d.difficulty;
        vis.xValue = d => d.time;
        vis.yValue = d => d.distance;

        // Set the scale input domains
        vis.xScale.domain([0, d3.max(vis.data, vis.xValue)]);
        vis.yScale.domain([0, d3.max(vis.data, vis.yValue)]);

        vis.renderVis();
    }

    protected renderVis() {
        let vis = this;
        vis.idValue = (d: any) => d.trail;

        // Bind data to circles
        const circles = vis.chart.selectAll('.point')
            .data(vis.data, (d: any) => vis.idValue(d));

        // Enter phase
        const circlesEnter = circles.enter()
            .append('circle')
            .attr('class', 'point')
            .attr('r', 4)
            .attr('fill', (d: any) => vis.colorScale(vis.colorValue(d)))
            .attr('cy', vis.yScale(0)) // Start from y=0
            .attr('cx', (d: any) => vis.xScale(vis.xValue(d)))
            .on('mouseover', (_event: any, d: any) => {
                vis.tooltip.style('display', 'block')
                    .html(`
                        <div class="tooltip-title">${d.trail}</div>
                        <div class="tooltip-second-title">${d.region}</div>
                        <ul>
                            <li>${d.distance} km, ~${d.time} hours</li>
                            <li>${d.difficulty}</li>
                            <li>${d.season}</li>
                        </ul>`
                    );
            })
            .on('mousemove', (event: any) => {
                vis.tooltip.style('left', (event.pageX + 10) + 'px')
                    .style('top', (event.pageY + 10) + 'px');
            })
            .on('mouseleave', () => {
                vis.tooltip.style('display', 'none');
            });

        circlesEnter.transition()
            .duration(1000)
            .attr('cy', (d: any) => vis.yScale(vis.yValue(d))); // Animate to the final position

        // Update phase
        circlesEnter.merge(circles)
            .transition()
            .duration(1000)
            .attr('cy', (d: any) => vis.yScale(vis.yValue(d)))
            .attr('cx', (d: any) => vis.xScale(vis.xValue(d)))
            .attr('fill', (d: any) => vis.colorScale(vis.colorValue(d)));

        // Exit phase: remove circles that no longer have data
        circles.exit()
            .transition()
            .duration(1000)
            .attr('r', 0)
            .remove();

        // Update the axes/gridlines
        vis.xAxisG
            .call(vis.xAxis)
            .call((g: any) => g.select('.domain').remove());

        vis.yAxisG
            .call(vis.yAxis)
            .call((g: any) => g.select('.domain').remove());
    }
}

