package usecase

import (
	"my-auth/models"
	"my-auth/repository"
	"my-auth/utils"
	"encoding/json"
	"errors"
	"log"
)

type userUsecase struct {
	userRepository repository.UserRepository
}

type UserUsecase interface {
	Register(username, password string) (bool, error)
	Login(username, password string) (string, error)
	ValidateToken(token string) (string, error)
}

func InitUserUsecase(userRepository repository.UserRepository) UserUsecase {
	return &userUsecase{
		userRepository,
	}
}

func (userUsecase *userUsecase) Register(username, password string) (bool, error) {
	hashedPassword, err := utils.HashPassword(password)
	if err != nil {
		log.Println("Error to register user: ", err)
		return false, err
	}

	_, err = userUsecase.userRepository.CreateUser(username, hashedPassword)
	if err != nil {
		log.Println("Error to register user: ", err)
		return false, err
	}

	return true, nil
}

func (userUsecase *userUsecase) Login(username, password string) (string, error) {
	user, err := userUsecase.userRepository.GetUserByUsername(username)
	if err != nil {
		log.Println("Error getting user by username:", err)
		return "", err
	}

	ok := utils.CheckPasswordHash(password, user.Password)
	if !ok {
		log.Println("Wrong password")
		return "", errors.New("Wrong password")
	}

	tokenString, err := utils.GenerateToken(models.User{
		ID:       user.ID,
		Username: user.Username,
	})
	if err != nil {
		log.Println("Error generating token:", err)
		return "", err
	}

	return tokenString, nil
}

func (userUsecase *userUsecase) ValidateToken(token string) (string, error) {
	user, err := utils.ParseToken(token)
	if err != nil {
		log.Println("Error validating token:", err)
		return "", errors.New("Invalid token. Please provide a valid token.")
	}

	result, err := json.Marshal(&user)
	if err != nil {
		log.Println("Error marshaling user data:", err)
		return "", errors.New("Error processing user data.")
	}

	return string(result), nil
}