import 'dart:io';

import 'package:flutter/material.dart';
import 'package:http/http.dart' as http;
import 'package:job_portal_app/Services/global_variables.dart';
import 'package:job_portal_app/data/Common/job.dart';

import '../Services/global_methods.dart';
import '../UI/Company/job_details_setting.dart';
import '../UI/Company/jobs_setting.dart';

class JobSettingWidget extends StatefulWidget {
  final Job job;

  const JobSettingWidget({
    super.key,
    required this.job,
  });

  @override
  State<JobSettingWidget> createState() => _JobSettingWidgetState();
}

class _JobSettingWidgetState extends State<JobSettingWidget> {
  @override
  Widget build(BuildContext context) {
    return Padding(
      padding: const EdgeInsets.all(8.0),
      child: Card(
        color: Colors.white24,
        child: ListTile(
          onLongPress: () {
            _deleteJob(widget.job.id);
          },
          onTap: () {
            Navigator.push(
                context,
                MaterialPageRoute(
                    builder: (context) =>
                        JobDetialsSetting(id: widget.job.id)));
          },
          leading: Container(
              width: 100,
              height: 150,
              padding: const EdgeInsets.only(right: 8.0),
              decoration: const BoxDecoration(
                border: Border(right: BorderSide(width: 1)),
              ),
              child: Expanded(
                child: Image.network(
                  '$myUrl/${widget.job.image}',
                  fit: BoxFit.cover,
                ),
              )),
          title: Text(widget.job.companyName,
              style: const TextStyle(
                  color: Colors.amber,
                  fontSize: 18,
                  fontWeight: FontWeight.bold),
              maxLines: 2,
              overflow: TextOverflow.ellipsis),
          subtitle: Column(
            mainAxisAlignment: MainAxisAlignment.start,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              Text(widget.job.name,
                  style: const TextStyle(
                      color: Colors.black,
                      fontSize: 15,
                      fontWeight: FontWeight.bold),
                  maxLines: 2,
                  overflow: TextOverflow.ellipsis),
              const SizedBox(
                height: 8,
              ),
              Text(widget.job.description,
                  style: const TextStyle(
                    color: Colors.black,
                    fontSize: 13,
                  ),
                  maxLines: 3,
                  overflow: TextOverflow.ellipsis),
              const SizedBox(
                height: 8,
              ),
              Row(
                mainAxisAlignment: MainAxisAlignment.start,
                children: [
                  Text(
                    widget.job.sallary.toString(),
                    style: const TextStyle(
                      color: Colors.black,
                      fontSize: 13,
                    ),
                  ),
                  const Icon(
                    Icons.attach_money,
                    size: 13,
                    color: Colors.black,
                  )
                ],
              )
            ],
          ),
          trailing: const Icon(
            Icons.keyboard_arrow_right,
            color: Colors.black,
            size: 30,
          ),
        ),
      ),
    );
  }

  void _deleteJob(int id) {
    try {
      final scaffoldMessenger = ScaffoldMessenger.of(context);

      showDialog(
        context: context,
        builder: (context) => AlertDialog(
          backgroundColor: Colors.black54,
          title: const Row(
            children: [
              Padding(
                padding: EdgeInsets.all(8.0),
                child: Icon(
                  Icons.logout,
                  size: 36,
                  color: Colors.white,
                ),
              ),
              Padding(
                padding: EdgeInsets.all(8.0),
                child: Text(
                  "Delete Job",
                  style: TextStyle(color: Colors.white, fontSize: 36),
                ),
              ),
            ],
          ),
          content: const Text(
            "Do you want to delete this Job?",
            style: TextStyle(color: Colors.white, fontSize: 20),
          ),
          actions: [
            TextButton(
                onPressed: () {
                  Navigator.canPop(context) ? Navigator.pop(context) : Null;
                },
                child: const Text(
                  "No",
                  style: TextStyle(color: Colors.green, fontSize: 16),
                )),
            TextButton(
              onPressed: () async {
                Navigator.pop(context);

                scaffoldMessenger.showSnackBar(const SnackBar(
                  content: Row(
                    mainAxisAlignment: MainAxisAlignment.spaceBetween,
                    children: [
                      Padding(
                        padding: EdgeInsets.all(5.0),
                        child: Text("Deleting..."),
                      ),
                      CircularProgressIndicator()
                    ],
                  ),
                  duration: Duration(seconds: 25),
                ));

                final response = await http.delete(
                    Uri.parse('$myUrl/api/Jobs/DeleteJob/$id'),
                    headers: {
                      HttpHeaders.authorizationHeader:
                          "Bearer ${identifier.token}"
                    });
                scaffoldMessenger.hideCurrentSnackBar();
                if (response.statusCode == 200) {
                  scaffoldMessenger.showSnackBar(const SnackBar(
                    content: Text('Job deleted successfully'),
                  ));
                  _popThenPush();
                } else {
                  scaffoldMessenger.showSnackBar(const SnackBar(
                    content: Text('Failed to delete job'),
                  ));
                }
              },
              child: const Text(
                "Yes",
                style: TextStyle(color: Colors.green, fontSize: 16),
              ),
            ),
          ],
        ),
      );
    } catch (error) {
      GlobalMethods.showErrorDialog(error: error.toString(), ctx: context);
    }
  }

  void _popThenPush() {
    Navigator.pushReplacement(
        context, MaterialPageRoute(builder: (context) => const JobsSetting()));
  }
}
