package Client;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpHeaders;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.concurrent.CompletableFuture;

public class WebClient {

    private HttpClient client;

    /* instantiate web client */
    /* Read more about Builder pattern https://en.wikipedia.org/wiki/Builder_pattern*/
    public WebClient() {
        this.client = HttpClient.newBuilder().version(HttpClient.Version.HTTP_2).build();
    }

    /* send task (post http request) asynchronously */
    public CompletableFuture<ResponseData> sendTask(String url, byte[] requestPayload) {
        HttpRequest request = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .POST(HttpRequest.BodyPublishers.ofByteArray(requestPayload))
                .build();

        return  client.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(response -> {
                    HttpHeaders responseHeaders = response.headers();
                    String body = response.body();
                    URI uri = response.uri();
                    String serverName = uri.getHost();
                    String port = String.valueOf(uri.getPort());
                    String debugInfo = responseHeaders.firstValue("X-Debug-Info").orElse("");
                    return new ResponseData(body,debugInfo, serverName+":"+port);
                });
    }

    /* send task (post http request) asynchronously with custom headers*/
    public CompletableFuture<ResponseData> sendTask(String url, byte[] requestPayload, String headers) {
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder()
                .uri(URI.create(url))
                .POST(HttpRequest.BodyPublishers.ofByteArray(requestPayload));

        if (headers != null && !headers.isEmpty()) {
            String[] headerLines = headers.split("\n");
            for (String headerLine : headerLines) {
                String[] headerParts = headerLine.split(":");
                String headerName = headerParts[0].trim();
                String headerValue = headerParts[1].trim();
                requestBuilder.header(headerName, headerValue);
            }
        }

        HttpRequest request = requestBuilder.build();

        return client.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(response -> {
                    HttpHeaders responseHeaders = response.headers();
                    String body = response.body();
                    URI uri = response.uri();
                    String serverName = uri.getHost();
                    String port = String.valueOf(uri.getPort());
                    String debugInfo = responseHeaders.firstValue("X-Debug-Info").orElse("");
                    return new ResponseData(body,debugInfo, serverName+":"+port);
                });
    }
}
