package org.example.driver;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import org.example.scanner.PortScanner;
import org.example.scanner.PortScannerResult;
import org.example.scanner.PortStatus;

/**
 * USAGE: JavaPortScanner [target IP] [port range]
 * <ul>
 * <li>[target IP]: IP address to scan</li>
 * <li>[port range]: range of ports to scan</li>
 * <ul>
 * <li>e.g. 80-100 scans from 80 to 100</li>
 * <li>e.g. 8080 scans only port 8080</li>
 * </ul>
 * </ul>
 */
public class Driver {

    private static final int SCAN_TIMEOUT = 200;
    private static final int THREAD_COUNT = Runtime.getRuntime().availableProcessors();

    public static void main(String[] args) {
        // get a list of Targets from Parser
        List<Target> targets = null;
        Scanner scanner = new Scanner(System.in);
        System.out.println("Enter <IPv4> or <IPv4 Range (startIP-endIP)>");
        String ipRange = scanner.nextLine();
        System.out.println("Enter [PortStart-PortEnd], if you want to scan all ports enter \"all\" word, else if you want to scan one port just enter port number ");
        String portRange = scanner.nextLine();
        scanner.close();
        ArrayList<String> arguments = new ArrayList<String>();

        arguments.add(ipRange);
        if (!portRange.equals("all")) {
            arguments.add(portRange);
        }
        try {
            targets = Parser.run(arguments);
        } catch (InvalidUserInputException e) {
            // exit with the message
            System.err.println(e.getMessage());
            System.exit(1);
        } catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
        // lists to populate
        List<List<Future<PortStatus>>> futureList = new ArrayList<List<Future<PortStatus>>>();
        List<List<PortStatus>> resultList = new ArrayList<List<PortStatus>>();

        // ExecutorService
        // loop through the targets
        for(int i=0; i<targets.size();i++){
            ExecutorService es = Executors.newFixedThreadPool(THREAD_COUNT);

            List<Future<PortStatus>> targetFutureList = new ArrayList<>();
            for (int currPort = targets.get(i).portRange[0]; currPort <= targets.get(i).portRange[1]; currPort++) {
                targetFutureList.add(PortScanner.isPortOpen(es, targets.get(i).targetIP, currPort, SCAN_TIMEOUT));
            }
            futureList.add(targetFutureList);
            es.shutdown();
            try {
                if (!es.awaitTermination(2, TimeUnit.MINUTES)) {
                    // timeout elapsed before termination
                    es.shutdownNow();
                }
            } catch (InterruptedException e) {
                es.shutdownNow();
            }
        }

        for(int i=0; i<futureList.size();i++){
            List<PortStatus> targetResultList = new ArrayList<>();
            // get results to resultList
            for (Future<PortStatus> eachFuture : futureList.get(i)) {
                try {
                    targetResultList.add(eachFuture.get(2, TimeUnit.MINUTES));
                } catch (Exception e) {
                    System.err.println("Timeout reached");
                    System.exit(1);
                }
            }
            resultList.add(targetResultList);
        }

        // display results
        List<PortScannerResult> portScannerResults = new ArrayList<>();
        for(int i=0; i<resultList.size();i++ ){
            PortScannerResult r = new PortScannerResult(resultList.get(i), resultList.get(i).get(0).getTargetIp());
            portScannerResults.add(r);
        }
        for(int i=0; i<portScannerResults.size();i++){
            System.out.println(portScannerResults.get(i));
        }
    }
}
