package org.example.driver;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;

public final class Parser {

    /**
     * Get target information from user inputs.
     *
     * @param args arguments from user
     */
    public static List<Target> run(ArrayList<String> args) throws InvalidUserInputException, UnknownHostException {
        List<Target> targets = new ArrayList<>();

        if (args.size() != 2 && args.size() != 1) {
            throw new InvalidUserInputException("default");
        }

        // set ip range
        String targetIPStart;
        String targetIPEnd;

        if (isValidIPv4Address(args.get(0))) {
            // single IP address provided
            targetIPStart = args.get(0);
            targetIPEnd = args.get(0);
        } else if (args.get(0).contains("-")) {
            // range of IP addresses

            // first occurrence of dash
            int index = args.get(0).indexOf("-");
            targetIPStart = args.get(0).substring(0, index);
            targetIPEnd = args.get(0).substring(index + 1, args.get(0).length());
        } else {
            // invalid IP address
            throw new InvalidUserInputException("ip");
        }

        long startIP = ipToLong(targetIPStart);
        long endIP = ipToLong(targetIPEnd);

        if (startIP > endIP) {
            throw new InvalidUserInputException("ip");
        }

        // set port range
        String targetPortStart;
        String targetPortEnd;

        if (args.size() == 1) {
            // target all possible port if provided IP addresses are valid
            targetPortStart = "1";
            targetPortEnd = "65535";
        } else if (args.get(1).contains("-")) {
            // is a range

            // first occurrence of dash
            int index = args.get(1).indexOf("-");
            targetPortStart = args.get(1).substring(0, index);
            targetPortEnd = args.get(1).substring(index + 1, args.get(1).length());
        } else {
            // a single port
            targetPortStart = args.get(1);
            targetPortEnd = args.get(1);
        }

        int startPort;
        int endPort;
        try {
            startPort = Integer.parseInt(targetPortStart);
            endPort = Integer.parseInt(targetPortEnd);

            // validate range
            if (startPort > endPort || !isValidPortNumber(startPort) || !isValidPortNumber(endPort)) {
                throw new InvalidUserInputException("port");
            }

        } catch (Exception e) {
            // could not convert port num to int
            throw new InvalidUserInputException("port");
        }

        // generate the list of targets
        for (long ip = startIP; ip <= endIP; ip++) {
            Target target = new Target();
            target.targetIP = longToIp(ip);
            target.portRange[0] = startPort;
            target.portRange[1] = endPort;
            targets.add(target);
        }

        return targets;
    }


    /**
     * Return true iff portNum is in valid range, 1 to 65535.
     *
     * @param portNum port number to check.
     * @return whether portNum is a valid port number.
     */
    private static boolean isValidPortNumber(int portNum) {
        return (1 <= portNum && portNum <= 65535);
    }

    /**
     * Validate provided IPv4 address.
     *
     * @param ip
     * @return whether ip is a valid IPv4 Address
     */
    private static boolean isValidIPv4Address(String ip) {
        try {
            return InetAddress.getByName(ip).getHostAddress().equals(ip);
        } catch (UnknownHostException e) {
            return false;
        }
    }
    private static long ipToLong(String ip) throws UnknownHostException {
        InetAddress address = InetAddress.getByName(ip);
        byte[] bytes = address.getAddress();

        long result = 0;
        for (byte b : bytes) {
            result <<= 8;
            result |= b & 0xFF;
        }
        return result;
    }
    private static String longToIp(long ip) {
        StringBuilder sb = new StringBuilder(15);
        for (int i = 0; i < 4; i++) {
            sb.insert(0, Long.toString(ip & 0xff));
            if (i < 3) {
                sb.insert(0, '.');
            }
            ip >>= 8;
        }
        return sb.toString();
    }
}
