package org.example;

import org.apache.zookeeper.*;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.data.Stat;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

public class LeaderElection implements Watcher {
    private static final String ADDRESS = "192.168.56.115:2181";
    private static final int SESSION_TIMEOUT = 3000;
    private static final String ELECTION_NAMESPACE = "/election";
    private String currentZnodeName;
    private String leader;
    private ZooKeeper zooKeeper = null;

    private void volunteerForLeadership() throws InterruptedException, KeeperException {
        String znodePrefix = ELECTION_NAMESPACE + "/c_";
        String path = zooKeeper.create(znodePrefix, new byte[]{}, ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL);
        this.currentZnodeName = path.replace(ELECTION_NAMESPACE + "/", "");
        System.out.println("Created znode: " + path);
    }

    private void electLeader() throws InterruptedException, KeeperException {
        List<String> children = zooKeeper.getChildren(ELECTION_NAMESPACE, false);
        Collections.sort(children);
        leader = children.get(0);
        if (currentZnodeName.equals(leader)) {
            System.out.println("I'm the leader");
            //set data in leader Znode.
            setLeaderData(currentZnodeName);
        } else {
            System.out.println("I'm not the leader, the leader is " + leader);
            // Set a watch on the znode with one less sequence number
            int currentZnodeIndex = children.indexOf(currentZnodeName);
            String previousZnode = ELECTION_NAMESPACE + "/" + children.get(currentZnodeIndex - 1);
            zooKeeper.exists(previousZnode, this);
        }
    }

    private void setLeaderData(String znodeName) {
        String znodePath = ELECTION_NAMESPACE + "/" + znodeName;
        try {
            zooKeeper.setData(znodePath, ("The leader is " + znodeName).getBytes(), -1);
        } catch (KeeperException | InterruptedException e) {
            throw new RuntimeException("Error setting leader data", e);
        }
    }


    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        LeaderElection leaderElection = new LeaderElection();
        leaderElection.connectToZookeeper();
        leaderElection.volunteerForLeadership();
        leaderElection.electLeader();
        leaderElection.run();
    }

    public void connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(ADDRESS, SESSION_TIMEOUT, this);
    }

    public void run() throws InterruptedException {
        synchronized (zooKeeper) {
            zooKeeper.wait();
        }
        zooKeeper.close();
    }

    @Override
    public void process(WatchedEvent watchedEvent) {
        switch (watchedEvent.getType()) {
            case None:
                handleNoneEvent(watchedEvent);
                break;
            case NodeDeleted:
                handleNodeDeletedEvent(watchedEvent);
                break;
            case NodeDataChanged:
                handleNodeDataChangedEvent(watchedEvent);
                break;
        }
    }

    private void handleNodeDeletedEvent(WatchedEvent watchedEvent) {
        String deletedZnodePath = watchedEvent.getPath();
        System.out.println("Znode deleted: " + deletedZnodePath);

        // Check if the deleted znode is the leader
        if (deletedZnodePath.equals(ELECTION_NAMESPACE + "/" + leader)) {
            System.out.println("The deleted znode was the leader. Re-electing leader...");

            try {
                // Re-elect leader
                electLeader();
            } catch (InterruptedException | KeeperException e) {
                throw new RuntimeException("Error re-electing leader", e);
            }
        }
    }


    private void handleNodeDataChangedEvent(WatchedEvent watchedEvent)  {
        String znodePath = watchedEvent.getPath();
        System.out.println("Znode data changed: " + znodePath);

        byte[] znodePathBytes = new byte[]{};
        try {
            //get data of the znode which changed its data
            znodePathBytes = zooKeeper.getData(znodePath, false, null);

            // Set the data of the current znode to the combined data
            zooKeeper.setData(ELECTION_NAMESPACE+"/"+currentZnodeName, znodePathBytes, -1);

            String data = new String(znodePathBytes);
            System.out.println(data);

        } catch (KeeperException e) {
            throw new RuntimeException(e);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

    }

    private void handleNoneEvent(WatchedEvent watchedEvent) {
        if (watchedEvent.getState() == Event.KeeperState.SyncConnected) {
            System.out.println("Successfully connected to ZooKeeper");
        } else if (watchedEvent.getState() == Event.KeeperState.Expired) {
            System.out.println("ZooKeeper session expired. Shutting down the program.");
            synchronized (zooKeeper) {
                zooKeeper.notifyAll();
            }
        } else if (watchedEvent.getState() == Event.KeeperState.Disconnected) {
            System.out.println("Disconnected from ZooKeeper. Shutting down the program.");
            synchronized (zooKeeper) {
                zooKeeper.notifyAll();
            }
        }
    }
}
