package com.example.UploadingFile_Client.Services;

import Uploading_Service.*;
import com.example.UploadingFile_Client.Constant;
import com.google.protobuf.ByteString;
import io.grpc.Metadata;
import io.grpc.stub.MetadataUtils;
import io.grpc.stub.StreamObserver;
import net.devh.boot.grpc.client.inject.GrpcClient;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.CountDownLatch;

@Service
public class UploadFileService {
   //Asynchronous stub of client
    private final uploadingServiceGrpc.uploadingServiceStub client;
    // @GrpcClient is help annotation refer to which server it requires to call
    //file-upload is the key will be read from properties file
    public UploadFileService(@GrpcClient(value = "file-upload") uploadingServiceGrpc.uploadingServiceStub client) {
        this.client = client;
    }
    public String uploadFile(final MultipartFile multipartFile) {
        //file name
        String name;
        //file size
        int size;
        //to read the bytes
        InputStream inputStream;
        name = multipartFile.getOriginalFilename();
        try {
            size = multipartFile.getBytes().length;
            inputStream = multipartFile.getInputStream();
        } catch (IOException e) {
            return "unable to extract file info";
        }
        StringBuilder  response = new StringBuilder();
        CountDownLatch countDownLatch = new CountDownLatch(1);

        Metadata metadata = new Metadata();
        metadata.put(Constant.fileMetaDataKey,
                FileMetadata.newBuilder()
                        .setFileNameWithType(name)
                        .setContentLength(size)
                        .build()
                        .toByteArray());
        //using this object we will stream the file content to the server
        StreamObserver<FileUploadRequest> fileUploadRequestStreamObserver = this.client
                .withInterceptors(MetadataUtils.newAttachHeadersInterceptor(metadata))
                .uploadFile(
                        new StreamObserver<FileUploadResponse>() {
                            @Override
                            //called when server sends the response
                            public void onNext(FileUploadResponse fileUploadResponse) {
                                response.append(fileUploadResponse.getUploadStatus());

                            }

                            @Override
                            //called when server send an error
                            public void onError(Throwable throwable) {
                                response.append(UploadStatus.FAILED);
                                throwable.printStackTrace();
                                countDownLatch.countDown();
                            }

                            @Override
                            //called when server finish processing the request
                            public void onCompleted() {
                                countDownLatch.countDown();
                            }
                        });
        //chunk data, i use the size 10 KB for each chunk
        byte[] tenKB = new byte[5120];

        int length;

        try {
            while ((length = inputStream.read(tenKB)) > 0) {
                FileUploadRequest request = FileUploadRequest
                        .newBuilder()
                        .setFile(File.newBuilder().setContent(ByteString.copyFrom(tenKB, 0, length)))
                        .build();
                //sending the request that contain the chunk data of file
                fileUploadRequestStreamObserver.onNext(request);
            }
            inputStream.close();
            //notify the server we have completed sending the chunk data
            fileUploadRequestStreamObserver.onCompleted();
            //waiting server response
            countDownLatch.await();

        } catch (Exception e) {
            e.printStackTrace();
            response.append(UploadStatus.FAILED);
        }
        return response.toString();
    }
}
