package HTTP;

import ch.qos.logback.classic.Logger;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import com.sun.net.httpserver.HttpServer;
import grpc.client.grpcClient;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

public class ServerHttp implements Watcher {
    private ZooKeeper zooKeeper;
    private static final Logger logger = (Logger) LoggerFactory.getLogger(ServerHttp.class);

    private static final String address = "192.168.56.115:2181";
    private static final int SESSION_TIMEOUT = 3000; //dead client
    private static grpcClient client;

    public static void main(String[] args) throws IOException, InterruptedException, KeeperException {
        ServerHttp serverHttp =new ServerHttp();
        ZooKeeper zooKeeper = serverHttp.connectToZookeeper();
         client= new grpcClient(zooKeeper);
        // Create a simple HTTP server listening on port 8080
        HttpServer server = HttpServer.create(new InetSocketAddress(7777), 0);

        // Create a context for the "/search" path
        server.createContext("/search", new SearchHandler());

        // Start the server
        server.start();

        System.out.println("Server started on port 7777");
    }

    public ZooKeeper connectToZookeeper() throws IOException {
        this.zooKeeper = new ZooKeeper(address, SESSION_TIMEOUT, this);
        return zooKeeper;
    }

    public void run() throws InterruptedException {
        synchronized (zooKeeper) {
            zooKeeper.wait();
        }
    }

    void close() throws InterruptedException {
        this.zooKeeper.close();
    }
    @Override
    public void process(WatchedEvent watchedEvent) {
        switch (watchedEvent.getType()) {
            case None:
                if (watchedEvent.getState() == Watcher.Event.KeeperState.SyncConnected) {
                    logger.info("Successfully connected to Zookeeper");
                } else if (watchedEvent.getState() == Watcher.Event.KeeperState.Disconnected) {
                    synchronized (zooKeeper) {
                        logger.error("Disconnected from Zookeeper");
                        zooKeeper.notifyAll();
                    }
                } else if (watchedEvent.getState() == Watcher.Event.KeeperState.Closed) {
                    logger.error ("Closed Successfully");
                }
                break;
        }
    }

    static class SearchHandler implements HttpHandler {
        @Override
        public void handle(HttpExchange exchange) throws IOException {
            // Get the current date and time
            LocalDateTime currentTime = LocalDateTime.now();

            // Format the date and time if needed
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
            String formattedTime = currentTime.format(formatter);

            System.out.println(formattedTime+"- receive request");
            TreeMap<String, String> queryParams = getQueryParams(exchange.getRequestURI());
            // Retrieve input string from query parameters
            String inputString = queryParams.get("inputString");

            // get sorted files with IDF values from a Coordinator
            String [] parts= inputString.split("_");
            String dataString="";
            for (String part:parts) {
                dataString+=part+" ";
            }
            Map<String, Double> FilesWithIDF= client.startClient(dataString) ;
            // Build JSON response
            String jsonResponse = mapToJson(FilesWithIDF);

            // Set response headers
            exchange.getResponseHeaders()
                    .set("Content-Type", "application/json");
            exchange.getResponseHeaders().set("Access-Control-Allow-Origin", "*");
            exchange.getResponseHeaders().set("Access-Control-Allow-Methods", "POST, GET, OPTIONS");
            exchange.getResponseHeaders().set("Access-Control-Allow-Headers", "Content-Type");



            exchange.sendResponseHeaders(200, jsonResponse.length());

            // Send response body
            try (OutputStream os = exchange.getResponseBody()) {
                os.write(jsonResponse.getBytes());
            }
        }
    }

    // Helper method to convert a Map to a JSON string
    private static String mapToJson(Map<String, Double> map) {
        StringBuilder json = new StringBuilder("{");
        for (Map.Entry<String, Double> entry : map.entrySet()) {
            json.append("\"").append(entry.getKey()).append("\":").append(entry.getValue()).append(",");
        }
        if (json.length() > 1) {
            json.setLength(json.length() - 1); // Remove trailing comma
        }
        json.append("}");
        return json.toString();
    }

    // Helper method to extract query parameters from a URI
    private static TreeMap<String, String> getQueryParams(java.net.URI uri) {
        TreeMap<String, String> result = new TreeMap<>();
        String query = uri.getQuery();
        if (query != null) {
            for (String param : query.split("&")) {
                String[] pair = param.split("=");
                if (pair.length > 1) {
                    try {
                        String key = java.net.URLDecoder.decode(pair[0], "UTF-8");
                        String value = java.net.URLDecoder.decode(pair[1], "UTF-8");
                        result.put(key, value);
                    } catch (java.io.UnsupportedEncodingException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        return result;
    }
}
