package Coordinator_Worker;

import java.io.File;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;

public class Worker {
    private static final int PORT = 4444;
    private String nodeName;
    private String ipAddress;
    private String userName;
    private static List<Document> documents = new ArrayList<>();

    public Worker(String fullPathNode) {
        this.nodeName = fullPathNode;
    }


    public static void startWorker() {
        // Step 1: Set up server socket to listen for connections
        while (true) {
            try {
                ServerSocket serverSocket = new ServerSocket(PORT);

                // Step 2: Accept connection from coordinator
                Socket coordinatorSocket = serverSocket.accept();

                // Step 3: Receive data from coordinator
                DataObject data = Communication.receiveData(coordinatorSocket);
                System.out.println("Received data from coordinator: " + data.getDataString() + ", " + data.getFileNames());

                // Step 4: Process the data using IDF algorithm
                List<Document> response = IDF(data);

                // Step 5: Send the result back to the coordinator
                Communication.sendResult(coordinatorSocket, response);

                // Step 6: Close sockets
                coordinatorSocket.close();
                serverSocket.close();
            } catch (IOException e) {
                e.printStackTrace();
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(e);
            }
        }
    }


    // Function to calculate IDF for each term in the search query across files
    private static List<Document> IDF(DataObject data) {
        String searchQuery = data.getDataString();
        List<Document> r=new ArrayList<>();
        String[] terms = searchQuery.split("\\s+");

        // Step 2: Use ExecutorService for parallel processing
        ExecutorService executorService = Executors.newFixedThreadPool(Runtime.getRuntime().availableProcessors());

        try {
            List<Callable<Document>> tasks=new ArrayList<>();
            List<Future<Document>> futures = new ArrayList<>();            // Step 3: Submit a task for each file to the ExecutorService
            for (String fileName : data.getFileNames()) {
                File file = new File("C:\\Users\\Abdullah\\Downloads\\network-communication-master\\Cordinator_Worker\\src\\main\\resources\\" + fileName);
                Callable<Document> task = () -> processFile(file, terms);
                tasks.add(task);
                futures.add(executorService.submit(task));
            }
            executorService.awaitTermination(2, TimeUnit.MILLISECONDS);
            // Step 4: Retrieve results from Future and merge into the 'r' list
            for (Future<Document> future : futures) {
                try {
                    Document document = future.get();
                    r.add(document);
                } catch (InterruptedException | ExecutionException e) {
                    e.printStackTrace();
                }
            }
        } catch (InterruptedException e) {
            e.printStackTrace();
        } finally {
            // Step 5: Shutdown the ExecutorService
            executorService.shutdown();
        }

        // Step 6: Return the final IDF map
        return r;
    }

    // Function to process each file and calculate term frequencies
    private static Document processFile(File file, String[] terms) {
        Document document = new Document(file.getName());
        int totalWords = 0;

        try {
            // Read content from the file
            String content = new String(Files.readAllBytes(file.toPath()));
            String[] words = content.split("\\s+");

            // Calculate term frequencies in the file
            for (String term : terms) {
                int frequency = 0;
                for (String word : words) {
                    if (word.equals(term)) {
                        frequency++;
                    }
                    totalWords++;
                }
                document.setTf(term, (double) (frequency / totalWords));
            }
        } catch (IOException e) {
            e.printStackTrace();
            // Handle file reading exception
        }finally {
            return document;
        }
    }

    private static synchronized void addToDocuments(Document document) {
        documents.add(document);
    }

    public void setIpAddress(String ipAddress) {
        this.ipAddress = ipAddress;
    }

    public void setUserName(String userName) {
        this.userName = userName;
    }

    public String getNodeName() {
        return nodeName;
    }

    public static List<String> getIpAddress(List<Worker> workers) {
        List<String> allAddress = new ArrayList<>();
        for (Worker worker : workers) {
            allAddress.add(worker.ipAddress);
        }
        return allAddress;
    }

    public String getIpAddress() {
        return ipAddress;
    }

    public String getUserName() {
        return userName;
    }
}
