package com.ali.notificationsvc.controllers;

import com.ali.notificationsvc.dtos.CreateNotificationDto;
import com.ali.notificationsvc.dtos.NotificationDto;
import com.ali.notificationsvc.services.INotificationService;
import com.ali.notificationsvc.status.RequestStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/api/notifications")
public class NotificationController {

    private static final Logger LOGGER = LoggerFactory.getLogger(NotificationController.class);

    @Autowired
    INotificationService notificationService;

    public NotificationController(INotificationService iNotificationService) {
        this.notificationService = iNotificationService;
    }
    @GetMapping("getById/{id}")
    public ResponseEntity<NotificationDto> getById(@PathVariable("id") int id) {
        LOGGER.info(String.format("get notification details by %s id",id));

        NotificationDto notificationDto = new NotificationDto();
        notificationDto = notificationService.getById(id);
        return ResponseEntity.ok(notificationDto);
    }
    @GetMapping("getAll")
    public ResponseEntity<List<NotificationDto>> getAllNotifications() {
        LOGGER.info(String.format("get all notification"));

        List<NotificationDto> notificationDtos = notificationService.getAll();
        return ResponseEntity.ok(notificationDtos);
    }
    @GetMapping("getAllByCompanyId/{id}")
    public ResponseEntity<List<NotificationDto>> getAllByCompanyId(@PathVariable("id") int companyId) {
        LOGGER.info(String.format("get all notification by company id: %s", companyId));

        List<NotificationDto> notificationDtos = notificationService.getNotificationByCompanyId(companyId);
        return ResponseEntity.ok(notificationDtos);
    }
    @GetMapping("getAllByJobseekerId/{id}")
    public ResponseEntity<List<NotificationDto>> getAllByJobseekerId(@PathVariable("id") int jobseekerId) {
        LOGGER.info(String.format("get all notification by jobseeker id: %s", jobseekerId));

        List<NotificationDto> notificationDtos = notificationService.getNotificationByJobseekerId(jobseekerId);
        return ResponseEntity.ok(notificationDtos);
    }
    @GetMapping("isExist/{id}")
    public ResponseEntity<Boolean> isExist(@PathVariable("id") int notificationId) {
        LOGGER.info(String.format("check if notification is exist where id: %s", notificationId));

        boolean isExist = notificationService.existById(notificationId);
        return ResponseEntity.ok(isExist);
    }
    @GetMapping("getUnReadNotificationNumForJobseeker/{id}")
    public ResponseEntity<Integer> getUnReadNotificationNumForJobseeker(@PathVariable("id") int jobseekerId) {
        LOGGER.info(String.format("get all un read notifications for jobseeker id: %s", jobseekerId));

        int num = notificationService.getUnReadNotificationsNumberByJobseekerId(jobseekerId);
        return ResponseEntity.ok(num);
    }
    @GetMapping("getUnReadNotificationNumForCompany/{id}")
    public ResponseEntity<Integer> getUnReadNotificationNumForCompany(@PathVariable("id") int companyId) {
        LOGGER.info(String.format("get all un read notifications for company id: %s", companyId));

        int num = notificationService.getUnReadNotificationsNumberByCompanyId(companyId);
        return ResponseEntity.ok(num);
    }

    @PostMapping("create")
    public ResponseEntity<RequestStatus> createNotification(@RequestBody CreateNotificationDto createNotificationDto) {
        LOGGER.info(String.format("create new notification in synchronized way"));

        RequestStatus status = notificationService.create(createNotificationDto);
        if (status.isStatus()) {
            return ResponseEntity.status(HttpStatus.CREATED).body(status);
        } else {
            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(status);
        }
    }
    @PutMapping("update")
    public ResponseEntity<RequestStatus> updateNotification(@RequestBody NotificationDto notificationDto) {
        LOGGER.info(String.format("update notification where id is: %s", notificationDto.getId()));

        RequestStatus status = notificationService.update(notificationDto);
        if (status.isStatus()) {
            return ResponseEntity.ok(status);
        } else {
            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(status);
        }
    }
    @DeleteMapping("delete/{id}")
    public ResponseEntity<RequestStatus> deleteNotification(@PathVariable("id") int id) {
        LOGGER.info(String.format("delete notification where id is: %s", id));

        RequestStatus status = notificationService.delete(id);
        if (status.isStatus()) {
            return ResponseEntity.ok(status);
        } else {
            return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(status);
        }
    }
}
