// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

// DO NOT EDIT -- DO NOT EDIT -- DO NOT EDIT
// This file is generated by dev/tools/gen_keycodes/bin/gen_keycodes.dart and
// should not be edited directly.
//
// Edit the template dev/tools/gen_keycodes/data/keyboard_key.tmpl instead.
// See dev/tools/gen_keycodes/README.md for more information.

import 'package:flutter/foundation.dart';

/// A base class for all keyboard key types.
///
/// See also:
///
///  * [PhysicalKeyboardKey], a class with static values that describe the keys
///    that are returned from [RawKeyEvent.physicalKey].
///  * [LogicalKeyboardKey], a class with static values that describe the keys
///    that are returned from [RawKeyEvent.logicalKey].
abstract class KeyboardKey extends Diagnosticable {
  /// A const constructor so that subclasses may be const.
  const KeyboardKey();
}

/// A class with static values that describe the keys that are returned from
/// [RawKeyEvent.logicalKey].
///
/// These represent *logical* keys, which are keys which are interpreted in the
/// context of any modifiers, modes, or keyboard layouts which may be in effect.
///
/// This is contrast to [PhysicalKeyboardKey], which represents a physical key
/// in a particular location on the keyboard, without regard for the modifier
/// state, mode, or keyboard layout.
///
/// As an example, if you wanted to implement an app where the "Q" key "quit"
/// something, you'd want to look at the logical key to detect this, since you
/// would like to have it match the key with "Q" on it, instead of always
/// looking for "the key next next to the TAB key", since on a French keyboard,
/// the key next to the TAB key has an "A" on it.
///
/// Conversely, if you wanted a game where the key next to the CAPS LOCK (the
/// "A" key on a QWERTY keyboard) moved the player to the left, you'd want to
/// look at the physical key to make sure that regardless of the character the
/// key produces, you got the key that is in that location on the keyboard.
///
/// {@tool sample --template=stateful_widget_scaffold}
/// This example shows how to detect if the user has selected the logical "Q"
/// key.
///
/// ```dart imports
/// import 'package:flutter/foundation.dart';
/// import 'package:flutter/services.dart';
/// ```
///
/// ```dart
/// // The node used to request the keyboard focus.
/// final FocusNode _focusNode = FocusNode();
/// // The message to display.
/// String _message;
///
/// // Focus nodes need to be disposed.
/// @override
/// void dispose() {
///   _focusNode.dispose();
///   super.dispose();
/// }
///
/// // Handles the key events from the RawKeyboardListener and update the
/// // _message.
/// void _handleKeyEvent(RawKeyEvent event) {
///   setState(() {
///     if (event.logicalKey == LogicalKeyboardKey.keyQ) {
///       _message = 'Pressed the "Q" key!';
///     } else {
///       if (kReleaseMode) {
///         _message = 'Not a Q: Key label is "${event.logicalKey.keyLabel ?? '<none>'}"';
///       } else {
///         // This will only print useful information in debug mode.
///         _message = 'Not a Q: Pressed ${event.logicalKey.debugName}';
///       }
///     }
///   });
/// }
///
/// @override
/// Widget build(BuildContext context) {
///   final TextTheme textTheme = Theme.of(context).textTheme;
///   return Container(
///     color: Colors.white,
///     alignment: Alignment.center,
///     child: DefaultTextStyle(
///       style: textTheme.display1,
///       child: RawKeyboardListener(
///         focusNode: _focusNode,
///         onKey: _handleKeyEvent,
///         child: AnimatedBuilder(
///           animation: _focusNode,
///           builder: (BuildContext context, Widget child) {
///             if (!_focusNode.hasFocus) {
///               return GestureDetector(
///                 onTap: () {
///                   FocusScope.of(context).requestFocus(_focusNode);
///                 },
///                 child: const Text('Tap to focus'),
///               );
///             }
///             return Text(_message ?? 'Press a key');
///           },
///         ),
///       ),
///     ),
///   );
/// }
/// ```
/// {@end-tool}
/// See also:
///
///  * [RawKeyEvent], the keyboard event object received by widgets that listen
///    to keyboard events.
///  * [RawKeyboardListener], a widget used to listen to and supply handlers for
///    keyboard events.
class LogicalKeyboardKey extends KeyboardKey {
  /// Creates a LogicalKeyboardKey object with an optional key label and debug
  /// name.
  ///
  /// [keyId] must not be null.
  ///
  /// {@tool snippet}
  /// To save executable size, it is recommended that the [debugName] be null in
  /// release mode. You can do this by using the [kReleaseMode] constant.
  ///
  /// ```dart
  /// const LogicalKeyboardKey(0x0010000000a, debugName: kReleaseMode ? null : 'Special Key')
  /// ```
  /// {@end-tool}
  const LogicalKeyboardKey(this.keyId, {this.debugName, this.keyLabel})
      : assert(keyId != null);

  /// A unique code representing this key.
  ///
  /// This is an opaque code. It should not be unpacked to derive information
  /// from it, as the representation of the code could change at any time.
  final int keyId;

  /// The debug string to print for this keyboard key, which will be null in
  /// release mode.
  final String debugName;

  /// The Unicode string representing the character produced by a [RawKeyEvent].
  ///
  /// This value is useful for describing or matching mnemonic keyboard
  /// shortcuts.
  ///
  /// On most platforms this is a single code point, but it could contain any
  /// Unicode string. The `keyLabel` differs from [RawKeyEvent.character]
  /// because `keyLabel` only takes into account the key being pressed, not any
  /// combining keys pressed before it, so, for example, an “o” that follows a
  /// combining dieresis (“¨”, COMBINING DIAERESIS (U+0308)) would just return
  /// “o” for [keyLabel], but would return “ö” for [RawKeyEvent.character].
  ///
  /// {@macro flutter.services.RawKeyEventData.keyLabel}
  final String keyLabel;

  @override
  int get hashCode => keyId.hashCode;

  @override
  bool operator ==(Object other) {
    if (other.runtimeType != runtimeType) {
      return false;
    }
    return other is LogicalKeyboardKey
        && other.keyId == keyId;
  }

  /// Returns the [LogicalKeyboardKey] constant that matches the given ID, or
  /// null, if not found.
  static LogicalKeyboardKey findKeyByKeyId(int keyId) => _knownLogicalKeys[keyId];

  /// Returns true if the given label represents a Unicode control character.
  ///
  /// Examples of control characters are characters like "U+000A LINE FEED (LF)"
  /// or "U+001B ESCAPE (ESC)".
  ///
  /// See <https://en.wikipedia.org/wiki/Unicode_control_characters> for more
  /// information.
  ///
  /// Used by [RawKeyEvent] subclasses to help construct IDs.
  static bool isControlCharacter(String label) {
    if (label.length > 1) {
      return false;
    }
    final int codeUnit = label.codeUnitAt(0);
    return (codeUnit <= 0x1f && codeUnit >= 0x00) || (codeUnit >= 0x7f && codeUnit <= 0x9f);
  }

  /// Returns true if the [keyId] of this object is one that is auto-generated by
  /// Flutter.
  ///
  /// Auto-generated key IDs are generated in response to platform key codes
  /// which Flutter doesn't recognize, and their IDs shouldn't be used in a
  /// persistent way.
  ///
  /// Auto-generated IDs should be a rare occurrence: Flutter supports most keys.
  ///
  /// Keys that generate Unicode characters (even if unknown to Flutter) will
  /// not return true for `isAutogenerated`, since they will be assigned a
  /// Unicode-based code that will remain stable.
  ///
  /// If Flutter adds support for a previously unsupported key code, the ID it
  /// reports will change, but the ID will remain stable on the platform it is
  /// produced on until Flutter adds support for recognizing it.
  ///
  /// So, hypothetically, if Android added a new key code of 0xffff,
  /// representing a new "do what I mean" key, then the auto-generated code
  /// would be 0x1020000ffff, but once Flutter added the "doWhatIMean" key to
  /// the definitions below, the new code would be 0x0020000ffff for all
  /// platforms that had a "do what I mean" key from then on.
  bool get isAutogenerated => (keyId & autogeneratedMask) != 0;

  /// Returns a set of pseudo-key synonyms for the given `key`.
  ///
  /// This allows finding the pseudo-keys that also represents a concrete
  /// `key` so that a class with a key map can match pseudo-keys as well as the
  /// actual generated keys.
  ///
  /// The pseudo-keys returned in the set are typically used to represent keys
  /// which appear in multiple places on the keyboard, such as the [shift],
  /// [alt], [control], and [meta] keys. The keys in the returned set won't ever
  /// be generated directly, but if a more specific key event is received, then
  /// this set can be used to find the more general pseudo-key. For example, if
  /// this is a [shiftLeft] key, this accessor will return the set
  /// `<LogicalKeyboardKey>{ shift }`.
  Set<LogicalKeyboardKey> get synonyms {
    final LogicalKeyboardKey result = _synonyms[this];
    return result == null ? <LogicalKeyboardKey>{} : <LogicalKeyboardKey>{result};
  }

  /// Takes a set of keys, and returns the same set, but with any keys that have
  /// synonyms replaced.
  ///
  /// It is used, for example, to make sets of keys with members like
  /// [controlRight] and [controlLeft] and convert that set to contain just
  /// [control], so that the question "is any control key down?" can be asked.
  static Set<LogicalKeyboardKey> collapseSynonyms(Set<LogicalKeyboardKey> input) {
    final Set<LogicalKeyboardKey> result = <LogicalKeyboardKey>{};
    for (final LogicalKeyboardKey key in input) {
      final LogicalKeyboardKey synonym = _synonyms[key];
      result.add(synonym ?? key);
    }
    return result;
  }

  @override
  void debugFillProperties(DiagnosticPropertiesBuilder properties) {
    super.debugFillProperties(properties);
    properties.add(StringProperty('keyId', '0x${keyId.toRadixString(16).padLeft(8, '0')}', showName: true));
    properties.add(StringProperty('keyLabel', keyLabel, showName: true));
    properties.add(StringProperty('debugName', debugName, showName: true, defaultValue: null));
  }

  /// Mask for the 32-bit value portion of the key code.
  ///
  /// This is used by platform-specific code to generate Flutter key codes.
  static const int valueMask = 0x000FFFFFFFF;

  /// Mask for the platform prefix portion of the key code.
  ///
  /// This is used by platform-specific code to generate Flutter key codes.
  static const int platformMask = 0x0FF00000000;

  /// Mask for the auto-generated bit portion of the key code.
  ///
  /// This is used by platform-specific code to generate new Flutter key codes
  /// for keys which are not recognized.
  static const int autogeneratedMask = 0x10000000000;

  /// Mask for the synonym pseudo-keys generated for keys which appear in more
  /// than one place on the keyboard.
  ///
  /// IDs in this range are used to represent keys which appear in multiple
  /// places on the keyboard, such as the SHIFT, ALT, CTRL, and numeric keypad
  /// keys. These key codes will never be generated by the key event system, but
  /// may be used in key maps to represent the union of all the keys of each
  /// type in order to match them.
  ///
  /// To look up the synonyms that are defined, look in the [synonyms] map.
  static const int synonymMask = 0x20000000000;

  /// The code prefix for keys which have a Unicode representation.
  ///
  /// This is used by platform-specific code to generate Flutter key codes.
  static const int unicodePlane = 0x00000000000;

  /// The code prefix for keys which do not have a Unicode representation.
  ///
  /// This is used by platform-specific code to generate Flutter key codes using
  /// HID Usage codes.
  static const int hidPlane = 0x00100000000;

  /// Represents the logical "None" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey none = LogicalKeyboardKey(0x00100000000, debugName: kReleaseMode ? null : 'None');

  /// Represents the logical "Hyper" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey hyper = LogicalKeyboardKey(0x00100000010, debugName: kReleaseMode ? null : 'Hyper');

  /// Represents the logical "Super Key" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey superKey = LogicalKeyboardKey(0x00100000011, debugName: kReleaseMode ? null : 'Super Key');

  /// Represents the logical "Fn Lock" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey fnLock = LogicalKeyboardKey(0x00100000013, debugName: kReleaseMode ? null : 'Fn Lock');

  /// Represents the logical "Suspend" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey suspend = LogicalKeyboardKey(0x00100000014, debugName: kReleaseMode ? null : 'Suspend');

  /// Represents the logical "Resume" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey resume = LogicalKeyboardKey(0x00100000015, debugName: kReleaseMode ? null : 'Resume');

  /// Represents the logical "Turbo" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey turbo = LogicalKeyboardKey(0x00100000016, debugName: kReleaseMode ? null : 'Turbo');

  /// Represents the logical "Sleep" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey sleep = LogicalKeyboardKey(0x00100010082, debugName: kReleaseMode ? null : 'Sleep');

  /// Represents the logical "Wake Up" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey wakeUp = LogicalKeyboardKey(0x00100010083, debugName: kReleaseMode ? null : 'Wake Up');

  /// Represents the logical "Display Toggle Int Ext" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey displayToggleIntExt = LogicalKeyboardKey(0x001000100b5, debugName: kReleaseMode ? null : 'Display Toggle Int Ext');

  /// Represents the logical "Usb Reserved" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey usbReserved = LogicalKeyboardKey(0x00100070000, debugName: kReleaseMode ? null : 'Usb Reserved');

  /// Represents the logical "Usb Error Roll Over" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey usbErrorRollOver = LogicalKeyboardKey(0x00100070001, debugName: kReleaseMode ? null : 'Usb Error Roll Over');

  /// Represents the logical "Usb Post Fail" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey usbPostFail = LogicalKeyboardKey(0x00100070002, debugName: kReleaseMode ? null : 'Usb Post Fail');

  /// Represents the logical "Usb Error Undefined" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey usbErrorUndefined = LogicalKeyboardKey(0x00100070003, debugName: kReleaseMode ? null : 'Usb Error Undefined');

  /// Represents the logical "Key A" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyA = LogicalKeyboardKey(0x00000000061, keyLabel: r'a', debugName: kReleaseMode ? null : 'Key A');

  /// Represents the logical "Key B" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyB = LogicalKeyboardKey(0x00000000062, keyLabel: r'b', debugName: kReleaseMode ? null : 'Key B');

  /// Represents the logical "Key C" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyC = LogicalKeyboardKey(0x00000000063, keyLabel: r'c', debugName: kReleaseMode ? null : 'Key C');

  /// Represents the logical "Key D" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyD = LogicalKeyboardKey(0x00000000064, keyLabel: r'd', debugName: kReleaseMode ? null : 'Key D');

  /// Represents the logical "Key E" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyE = LogicalKeyboardKey(0x00000000065, keyLabel: r'e', debugName: kReleaseMode ? null : 'Key E');

  /// Represents the logical "Key F" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyF = LogicalKeyboardKey(0x00000000066, keyLabel: r'f', debugName: kReleaseMode ? null : 'Key F');

  /// Represents the logical "Key G" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyG = LogicalKeyboardKey(0x00000000067, keyLabel: r'g', debugName: kReleaseMode ? null : 'Key G');

  /// Represents the logical "Key H" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyH = LogicalKeyboardKey(0x00000000068, keyLabel: r'h', debugName: kReleaseMode ? null : 'Key H');

  /// Represents the logical "Key I" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyI = LogicalKeyboardKey(0x00000000069, keyLabel: r'i', debugName: kReleaseMode ? null : 'Key I');

  /// Represents the logical "Key J" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyJ = LogicalKeyboardKey(0x0000000006a, keyLabel: r'j', debugName: kReleaseMode ? null : 'Key J');

  /// Represents the logical "Key K" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyK = LogicalKeyboardKey(0x0000000006b, keyLabel: r'k', debugName: kReleaseMode ? null : 'Key K');

  /// Represents the logical "Key L" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyL = LogicalKeyboardKey(0x0000000006c, keyLabel: r'l', debugName: kReleaseMode ? null : 'Key L');

  /// Represents the logical "Key M" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyM = LogicalKeyboardKey(0x0000000006d, keyLabel: r'm', debugName: kReleaseMode ? null : 'Key M');

  /// Represents the logical "Key N" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyN = LogicalKeyboardKey(0x0000000006e, keyLabel: r'n', debugName: kReleaseMode ? null : 'Key N');

  /// Represents the logical "Key O" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyO = LogicalKeyboardKey(0x0000000006f, keyLabel: r'o', debugName: kReleaseMode ? null : 'Key O');

  /// Represents the logical "Key P" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyP = LogicalKeyboardKey(0x00000000070, keyLabel: r'p', debugName: kReleaseMode ? null : 'Key P');

  /// Represents the logical "Key Q" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyQ = LogicalKeyboardKey(0x00000000071, keyLabel: r'q', debugName: kReleaseMode ? null : 'Key Q');

  /// Represents the logical "Key R" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyR = LogicalKeyboardKey(0x00000000072, keyLabel: r'r', debugName: kReleaseMode ? null : 'Key R');

  /// Represents the logical "Key S" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyS = LogicalKeyboardKey(0x00000000073, keyLabel: r's', debugName: kReleaseMode ? null : 'Key S');

  /// Represents the logical "Key T" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyT = LogicalKeyboardKey(0x00000000074, keyLabel: r't', debugName: kReleaseMode ? null : 'Key T');

  /// Represents the logical "Key U" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyU = LogicalKeyboardKey(0x00000000075, keyLabel: r'u', debugName: kReleaseMode ? null : 'Key U');

  /// Represents the logical "Key V" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyV = LogicalKeyboardKey(0x00000000076, keyLabel: r'v', debugName: kReleaseMode ? null : 'Key V');

  /// Represents the logical "Key W" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyW = LogicalKeyboardKey(0x00000000077, keyLabel: r'w', debugName: kReleaseMode ? null : 'Key W');

  /// Represents the logical "Key X" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyX = LogicalKeyboardKey(0x00000000078, keyLabel: r'x', debugName: kReleaseMode ? null : 'Key X');

  /// Represents the logical "Key Y" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyY = LogicalKeyboardKey(0x00000000079, keyLabel: r'y', debugName: kReleaseMode ? null : 'Key Y');

  /// Represents the logical "Key Z" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyZ = LogicalKeyboardKey(0x0000000007a, keyLabel: r'z', debugName: kReleaseMode ? null : 'Key Z');

  /// Represents the logical "Digit 1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit1 = LogicalKeyboardKey(0x00000000031, keyLabel: r'1', debugName: kReleaseMode ? null : 'Digit 1');

  /// Represents the logical "Digit 2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit2 = LogicalKeyboardKey(0x00000000032, keyLabel: r'2', debugName: kReleaseMode ? null : 'Digit 2');

  /// Represents the logical "Digit 3" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit3 = LogicalKeyboardKey(0x00000000033, keyLabel: r'3', debugName: kReleaseMode ? null : 'Digit 3');

  /// Represents the logical "Digit 4" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit4 = LogicalKeyboardKey(0x00000000034, keyLabel: r'4', debugName: kReleaseMode ? null : 'Digit 4');

  /// Represents the logical "Digit 5" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit5 = LogicalKeyboardKey(0x00000000035, keyLabel: r'5', debugName: kReleaseMode ? null : 'Digit 5');

  /// Represents the logical "Digit 6" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit6 = LogicalKeyboardKey(0x00000000036, keyLabel: r'6', debugName: kReleaseMode ? null : 'Digit 6');

  /// Represents the logical "Digit 7" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit7 = LogicalKeyboardKey(0x00000000037, keyLabel: r'7', debugName: kReleaseMode ? null : 'Digit 7');

  /// Represents the logical "Digit 8" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit8 = LogicalKeyboardKey(0x00000000038, keyLabel: r'8', debugName: kReleaseMode ? null : 'Digit 8');

  /// Represents the logical "Digit 9" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit9 = LogicalKeyboardKey(0x00000000039, keyLabel: r'9', debugName: kReleaseMode ? null : 'Digit 9');

  /// Represents the logical "Digit 0" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey digit0 = LogicalKeyboardKey(0x00000000030, keyLabel: r'0', debugName: kReleaseMode ? null : 'Digit 0');

  /// Represents the logical "Enter" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey enter = LogicalKeyboardKey(0x00100070028, debugName: kReleaseMode ? null : 'Enter');

  /// Represents the logical "Escape" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey escape = LogicalKeyboardKey(0x00100070029, debugName: kReleaseMode ? null : 'Escape');

  /// Represents the logical "Backspace" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey backspace = LogicalKeyboardKey(0x0010007002a, debugName: kReleaseMode ? null : 'Backspace');

  /// Represents the logical "Tab" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey tab = LogicalKeyboardKey(0x0010007002b, debugName: kReleaseMode ? null : 'Tab');

  /// Represents the logical "Space" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey space = LogicalKeyboardKey(0x00000000020, keyLabel: r' ', debugName: kReleaseMode ? null : 'Space');

  /// Represents the logical "Minus" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey minus = LogicalKeyboardKey(0x0000000002d, keyLabel: r'-', debugName: kReleaseMode ? null : 'Minus');

  /// Represents the logical "Equal" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey equal = LogicalKeyboardKey(0x0000000003d, keyLabel: r'=', debugName: kReleaseMode ? null : 'Equal');

  /// Represents the logical "Bracket Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey bracketLeft = LogicalKeyboardKey(0x0000000005b, keyLabel: r'[', debugName: kReleaseMode ? null : 'Bracket Left');

  /// Represents the logical "Bracket Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey bracketRight = LogicalKeyboardKey(0x0000000005d, keyLabel: r']', debugName: kReleaseMode ? null : 'Bracket Right');

  /// Represents the logical "Backslash" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey backslash = LogicalKeyboardKey(0x0000000005c, keyLabel: r'\', debugName: kReleaseMode ? null : 'Backslash');

  /// Represents the logical "Semicolon" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey semicolon = LogicalKeyboardKey(0x0000000003b, keyLabel: r';', debugName: kReleaseMode ? null : 'Semicolon');

  /// Represents the logical "Quote" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey quote = LogicalKeyboardKey(0x00000000027, keyLabel: r"'", debugName: kReleaseMode ? null : 'Quote');

  /// Represents the logical "Backquote" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey backquote = LogicalKeyboardKey(0x00000000060, keyLabel: r'`', debugName: kReleaseMode ? null : 'Backquote');

  /// Represents the logical "Comma" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey comma = LogicalKeyboardKey(0x0000000002c, keyLabel: r',', debugName: kReleaseMode ? null : 'Comma');

  /// Represents the logical "Period" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey period = LogicalKeyboardKey(0x0000000002e, keyLabel: r'.', debugName: kReleaseMode ? null : 'Period');

  /// Represents the logical "Slash" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey slash = LogicalKeyboardKey(0x0000000002f, keyLabel: r'/', debugName: kReleaseMode ? null : 'Slash');

  /// Represents the logical "Caps Lock" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey capsLock = LogicalKeyboardKey(0x00100070039, debugName: kReleaseMode ? null : 'Caps Lock');

  /// Represents the logical "F1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f1 = LogicalKeyboardKey(0x0010007003a, debugName: kReleaseMode ? null : 'F1');

  /// Represents the logical "F2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f2 = LogicalKeyboardKey(0x0010007003b, debugName: kReleaseMode ? null : 'F2');

  /// Represents the logical "F3" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f3 = LogicalKeyboardKey(0x0010007003c, debugName: kReleaseMode ? null : 'F3');

  /// Represents the logical "F4" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f4 = LogicalKeyboardKey(0x0010007003d, debugName: kReleaseMode ? null : 'F4');

  /// Represents the logical "F5" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f5 = LogicalKeyboardKey(0x0010007003e, debugName: kReleaseMode ? null : 'F5');

  /// Represents the logical "F6" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f6 = LogicalKeyboardKey(0x0010007003f, debugName: kReleaseMode ? null : 'F6');

  /// Represents the logical "F7" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f7 = LogicalKeyboardKey(0x00100070040, debugName: kReleaseMode ? null : 'F7');

  /// Represents the logical "F8" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f8 = LogicalKeyboardKey(0x00100070041, debugName: kReleaseMode ? null : 'F8');

  /// Represents the logical "F9" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f9 = LogicalKeyboardKey(0x00100070042, debugName: kReleaseMode ? null : 'F9');

  /// Represents the logical "F10" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f10 = LogicalKeyboardKey(0x00100070043, debugName: kReleaseMode ? null : 'F10');

  /// Represents the logical "F11" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f11 = LogicalKeyboardKey(0x00100070044, debugName: kReleaseMode ? null : 'F11');

  /// Represents the logical "F12" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f12 = LogicalKeyboardKey(0x00100070045, debugName: kReleaseMode ? null : 'F12');

  /// Represents the logical "Print Screen" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey printScreen = LogicalKeyboardKey(0x00100070046, debugName: kReleaseMode ? null : 'Print Screen');

  /// Represents the logical "Scroll Lock" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey scrollLock = LogicalKeyboardKey(0x00100070047, debugName: kReleaseMode ? null : 'Scroll Lock');

  /// Represents the logical "Pause" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey pause = LogicalKeyboardKey(0x00100070048, debugName: kReleaseMode ? null : 'Pause');

  /// Represents the logical "Insert" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey insert = LogicalKeyboardKey(0x00100070049, debugName: kReleaseMode ? null : 'Insert');

  /// Represents the logical "Home" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey home = LogicalKeyboardKey(0x0010007004a, debugName: kReleaseMode ? null : 'Home');

  /// Represents the logical "Page Up" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey pageUp = LogicalKeyboardKey(0x0010007004b, debugName: kReleaseMode ? null : 'Page Up');

  /// Represents the logical "Delete" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey delete = LogicalKeyboardKey(0x0010007004c, debugName: kReleaseMode ? null : 'Delete');

  /// Represents the logical "End" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey end = LogicalKeyboardKey(0x0010007004d, debugName: kReleaseMode ? null : 'End');

  /// Represents the logical "Page Down" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey pageDown = LogicalKeyboardKey(0x0010007004e, debugName: kReleaseMode ? null : 'Page Down');

  /// Represents the logical "Arrow Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey arrowRight = LogicalKeyboardKey(0x0010007004f, debugName: kReleaseMode ? null : 'Arrow Right');

  /// Represents the logical "Arrow Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey arrowLeft = LogicalKeyboardKey(0x00100070050, debugName: kReleaseMode ? null : 'Arrow Left');

  /// Represents the logical "Arrow Down" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey arrowDown = LogicalKeyboardKey(0x00100070051, debugName: kReleaseMode ? null : 'Arrow Down');

  /// Represents the logical "Arrow Up" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey arrowUp = LogicalKeyboardKey(0x00100070052, debugName: kReleaseMode ? null : 'Arrow Up');

  /// Represents the logical "Num Lock" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numLock = LogicalKeyboardKey(0x00100070053, debugName: kReleaseMode ? null : 'Num Lock');

  /// Represents the logical "Numpad Divide" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadDivide = LogicalKeyboardKey(0x00100070054, keyLabel: r'/', debugName: kReleaseMode ? null : 'Numpad Divide');

  /// Represents the logical "Numpad Multiply" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadMultiply = LogicalKeyboardKey(0x00100070055, keyLabel: r'*', debugName: kReleaseMode ? null : 'Numpad Multiply');

  /// Represents the logical "Numpad Subtract" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadSubtract = LogicalKeyboardKey(0x00100070056, keyLabel: r'-', debugName: kReleaseMode ? null : 'Numpad Subtract');

  /// Represents the logical "Numpad Add" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadAdd = LogicalKeyboardKey(0x00100070057, keyLabel: r'+', debugName: kReleaseMode ? null : 'Numpad Add');

  /// Represents the logical "Numpad Enter" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadEnter = LogicalKeyboardKey(0x00100070058, debugName: kReleaseMode ? null : 'Numpad Enter');

  /// Represents the logical "Numpad 1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad1 = LogicalKeyboardKey(0x00100070059, keyLabel: r'1', debugName: kReleaseMode ? null : 'Numpad 1');

  /// Represents the logical "Numpad 2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad2 = LogicalKeyboardKey(0x0010007005a, keyLabel: r'2', debugName: kReleaseMode ? null : 'Numpad 2');

  /// Represents the logical "Numpad 3" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad3 = LogicalKeyboardKey(0x0010007005b, keyLabel: r'3', debugName: kReleaseMode ? null : 'Numpad 3');

  /// Represents the logical "Numpad 4" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad4 = LogicalKeyboardKey(0x0010007005c, keyLabel: r'4', debugName: kReleaseMode ? null : 'Numpad 4');

  /// Represents the logical "Numpad 5" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad5 = LogicalKeyboardKey(0x0010007005d, keyLabel: r'5', debugName: kReleaseMode ? null : 'Numpad 5');

  /// Represents the logical "Numpad 6" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad6 = LogicalKeyboardKey(0x0010007005e, keyLabel: r'6', debugName: kReleaseMode ? null : 'Numpad 6');

  /// Represents the logical "Numpad 7" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad7 = LogicalKeyboardKey(0x0010007005f, keyLabel: r'7', debugName: kReleaseMode ? null : 'Numpad 7');

  /// Represents the logical "Numpad 8" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad8 = LogicalKeyboardKey(0x00100070060, keyLabel: r'8', debugName: kReleaseMode ? null : 'Numpad 8');

  /// Represents the logical "Numpad 9" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad9 = LogicalKeyboardKey(0x00100070061, keyLabel: r'9', debugName: kReleaseMode ? null : 'Numpad 9');

  /// Represents the logical "Numpad 0" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpad0 = LogicalKeyboardKey(0x00100070062, keyLabel: r'0', debugName: kReleaseMode ? null : 'Numpad 0');

  /// Represents the logical "Numpad Decimal" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadDecimal = LogicalKeyboardKey(0x00100070063, keyLabel: r'.', debugName: kReleaseMode ? null : 'Numpad Decimal');

  /// Represents the logical "Intl Backslash" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey intlBackslash = LogicalKeyboardKey(0x00100070064, debugName: kReleaseMode ? null : 'Intl Backslash');

  /// Represents the logical "Context Menu" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey contextMenu = LogicalKeyboardKey(0x00100070065, debugName: kReleaseMode ? null : 'Context Menu');

  /// Represents the logical "Power" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey power = LogicalKeyboardKey(0x00100070066, debugName: kReleaseMode ? null : 'Power');

  /// Represents the logical "Numpad Equal" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadEqual = LogicalKeyboardKey(0x00100070067, keyLabel: r'=', debugName: kReleaseMode ? null : 'Numpad Equal');

  /// Represents the logical "F13" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f13 = LogicalKeyboardKey(0x00100070068, debugName: kReleaseMode ? null : 'F13');

  /// Represents the logical "F14" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f14 = LogicalKeyboardKey(0x00100070069, debugName: kReleaseMode ? null : 'F14');

  /// Represents the logical "F15" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f15 = LogicalKeyboardKey(0x0010007006a, debugName: kReleaseMode ? null : 'F15');

  /// Represents the logical "F16" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f16 = LogicalKeyboardKey(0x0010007006b, debugName: kReleaseMode ? null : 'F16');

  /// Represents the logical "F17" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f17 = LogicalKeyboardKey(0x0010007006c, debugName: kReleaseMode ? null : 'F17');

  /// Represents the logical "F18" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f18 = LogicalKeyboardKey(0x0010007006d, debugName: kReleaseMode ? null : 'F18');

  /// Represents the logical "F19" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f19 = LogicalKeyboardKey(0x0010007006e, debugName: kReleaseMode ? null : 'F19');

  /// Represents the logical "F20" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f20 = LogicalKeyboardKey(0x0010007006f, debugName: kReleaseMode ? null : 'F20');

  /// Represents the logical "F21" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f21 = LogicalKeyboardKey(0x00100070070, debugName: kReleaseMode ? null : 'F21');

  /// Represents the logical "F22" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f22 = LogicalKeyboardKey(0x00100070071, debugName: kReleaseMode ? null : 'F22');

  /// Represents the logical "F23" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f23 = LogicalKeyboardKey(0x00100070072, debugName: kReleaseMode ? null : 'F23');

  /// Represents the logical "F24" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey f24 = LogicalKeyboardKey(0x00100070073, debugName: kReleaseMode ? null : 'F24');

  /// Represents the logical "Open" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey open = LogicalKeyboardKey(0x00100070074, debugName: kReleaseMode ? null : 'Open');

  /// Represents the logical "Help" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey help = LogicalKeyboardKey(0x00100070075, debugName: kReleaseMode ? null : 'Help');

  /// Represents the logical "Select" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey select = LogicalKeyboardKey(0x00100070077, debugName: kReleaseMode ? null : 'Select');

  /// Represents the logical "Again" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey again = LogicalKeyboardKey(0x00100070079, debugName: kReleaseMode ? null : 'Again');

  /// Represents the logical "Undo" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey undo = LogicalKeyboardKey(0x0010007007a, debugName: kReleaseMode ? null : 'Undo');

  /// Represents the logical "Cut" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey cut = LogicalKeyboardKey(0x0010007007b, debugName: kReleaseMode ? null : 'Cut');

  /// Represents the logical "Copy" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey copy = LogicalKeyboardKey(0x0010007007c, debugName: kReleaseMode ? null : 'Copy');

  /// Represents the logical "Paste" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey paste = LogicalKeyboardKey(0x0010007007d, debugName: kReleaseMode ? null : 'Paste');

  /// Represents the logical "Find" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey find = LogicalKeyboardKey(0x0010007007e, debugName: kReleaseMode ? null : 'Find');

  /// Represents the logical "Audio Volume Mute" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey audioVolumeMute = LogicalKeyboardKey(0x0010007007f, debugName: kReleaseMode ? null : 'Audio Volume Mute');

  /// Represents the logical "Audio Volume Up" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey audioVolumeUp = LogicalKeyboardKey(0x00100070080, debugName: kReleaseMode ? null : 'Audio Volume Up');

  /// Represents the logical "Audio Volume Down" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey audioVolumeDown = LogicalKeyboardKey(0x00100070081, debugName: kReleaseMode ? null : 'Audio Volume Down');

  /// Represents the logical "Numpad Comma" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadComma = LogicalKeyboardKey(0x00100070085, keyLabel: r',', debugName: kReleaseMode ? null : 'Numpad Comma');

  /// Represents the logical "Intl Ro" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey intlRo = LogicalKeyboardKey(0x00100070087, debugName: kReleaseMode ? null : 'Intl Ro');

  /// Represents the logical "Kana Mode" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey kanaMode = LogicalKeyboardKey(0x00100070088, debugName: kReleaseMode ? null : 'Kana Mode');

  /// Represents the logical "Intl Yen" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey intlYen = LogicalKeyboardKey(0x00100070089, debugName: kReleaseMode ? null : 'Intl Yen');

  /// Represents the logical "Convert" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey convert = LogicalKeyboardKey(0x0010007008a, debugName: kReleaseMode ? null : 'Convert');

  /// Represents the logical "Non Convert" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey nonConvert = LogicalKeyboardKey(0x0010007008b, debugName: kReleaseMode ? null : 'Non Convert');

  /// Represents the logical "Lang 1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey lang1 = LogicalKeyboardKey(0x00100070090, debugName: kReleaseMode ? null : 'Lang 1');

  /// Represents the logical "Lang 2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey lang2 = LogicalKeyboardKey(0x00100070091, debugName: kReleaseMode ? null : 'Lang 2');

  /// Represents the logical "Lang 3" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey lang3 = LogicalKeyboardKey(0x00100070092, debugName: kReleaseMode ? null : 'Lang 3');

  /// Represents the logical "Lang 4" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey lang4 = LogicalKeyboardKey(0x00100070093, debugName: kReleaseMode ? null : 'Lang 4');

  /// Represents the logical "Lang 5" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey lang5 = LogicalKeyboardKey(0x00100070094, debugName: kReleaseMode ? null : 'Lang 5');

  /// Represents the logical "Abort" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey abort = LogicalKeyboardKey(0x0010007009b, debugName: kReleaseMode ? null : 'Abort');

  /// Represents the logical "Props" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey props = LogicalKeyboardKey(0x001000700a3, debugName: kReleaseMode ? null : 'Props');

  /// Represents the logical "Numpad Paren Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadParenLeft = LogicalKeyboardKey(0x001000700b6, keyLabel: r'(', debugName: kReleaseMode ? null : 'Numpad Paren Left');

  /// Represents the logical "Numpad Paren Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadParenRight = LogicalKeyboardKey(0x001000700b7, keyLabel: r')', debugName: kReleaseMode ? null : 'Numpad Paren Right');

  /// Represents the logical "Numpad Backspace" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadBackspace = LogicalKeyboardKey(0x001000700bb, debugName: kReleaseMode ? null : 'Numpad Backspace');

  /// Represents the logical "Numpad Memory Store" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadMemoryStore = LogicalKeyboardKey(0x001000700d0, debugName: kReleaseMode ? null : 'Numpad Memory Store');

  /// Represents the logical "Numpad Memory Recall" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadMemoryRecall = LogicalKeyboardKey(0x001000700d1, debugName: kReleaseMode ? null : 'Numpad Memory Recall');

  /// Represents the logical "Numpad Memory Clear" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadMemoryClear = LogicalKeyboardKey(0x001000700d2, debugName: kReleaseMode ? null : 'Numpad Memory Clear');

  /// Represents the logical "Numpad Memory Add" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadMemoryAdd = LogicalKeyboardKey(0x001000700d3, debugName: kReleaseMode ? null : 'Numpad Memory Add');

  /// Represents the logical "Numpad Memory Subtract" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadMemorySubtract = LogicalKeyboardKey(0x001000700d4, debugName: kReleaseMode ? null : 'Numpad Memory Subtract');

  /// Represents the logical "Numpad Sign Change" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadSignChange = LogicalKeyboardKey(0x001000700d7, debugName: kReleaseMode ? null : 'Numpad Sign Change');

  /// Represents the logical "Numpad Clear" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadClear = LogicalKeyboardKey(0x001000700d8, debugName: kReleaseMode ? null : 'Numpad Clear');

  /// Represents the logical "Numpad Clear Entry" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey numpadClearEntry = LogicalKeyboardKey(0x001000700d9, debugName: kReleaseMode ? null : 'Numpad Clear Entry');

  /// Represents the logical "Control Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey controlLeft = LogicalKeyboardKey(0x001000700e0, debugName: kReleaseMode ? null : 'Control Left');

  /// Represents the logical "Shift Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey shiftLeft = LogicalKeyboardKey(0x001000700e1, debugName: kReleaseMode ? null : 'Shift Left');

  /// Represents the logical "Alt Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey altLeft = LogicalKeyboardKey(0x001000700e2, debugName: kReleaseMode ? null : 'Alt Left');

  /// Represents the logical "Meta Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey metaLeft = LogicalKeyboardKey(0x001000700e3, debugName: kReleaseMode ? null : 'Meta Left');

  /// Represents the logical "Control Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey controlRight = LogicalKeyboardKey(0x001000700e4, debugName: kReleaseMode ? null : 'Control Right');

  /// Represents the logical "Shift Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey shiftRight = LogicalKeyboardKey(0x001000700e5, debugName: kReleaseMode ? null : 'Shift Right');

  /// Represents the logical "Alt Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey altRight = LogicalKeyboardKey(0x001000700e6, debugName: kReleaseMode ? null : 'Alt Right');

  /// Represents the logical "Meta Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey metaRight = LogicalKeyboardKey(0x001000700e7, debugName: kReleaseMode ? null : 'Meta Right');

  /// Represents the logical "Info" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey info = LogicalKeyboardKey(0x001000c0060, debugName: kReleaseMode ? null : 'Info');

  /// Represents the logical "Closed Caption Toggle" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey closedCaptionToggle = LogicalKeyboardKey(0x001000c0061, debugName: kReleaseMode ? null : 'Closed Caption Toggle');

  /// Represents the logical "Brightness Up" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey brightnessUp = LogicalKeyboardKey(0x001000c006f, debugName: kReleaseMode ? null : 'Brightness Up');

  /// Represents the logical "Brightness Down" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey brightnessDown = LogicalKeyboardKey(0x001000c0070, debugName: kReleaseMode ? null : 'Brightness Down');

  /// Represents the logical "Brightness Toggle" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey brightnessToggle = LogicalKeyboardKey(0x001000c0072, debugName: kReleaseMode ? null : 'Brightness Toggle');

  /// Represents the logical "Brightness Minimum" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey brightnessMinimum = LogicalKeyboardKey(0x001000c0073, debugName: kReleaseMode ? null : 'Brightness Minimum');

  /// Represents the logical "Brightness Maximum" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey brightnessMaximum = LogicalKeyboardKey(0x001000c0074, debugName: kReleaseMode ? null : 'Brightness Maximum');

  /// Represents the logical "Brightness Auto" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey brightnessAuto = LogicalKeyboardKey(0x001000c0075, debugName: kReleaseMode ? null : 'Brightness Auto');

  /// Represents the logical "Media Last" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaLast = LogicalKeyboardKey(0x001000c0083, debugName: kReleaseMode ? null : 'Media Last');

  /// Represents the logical "Launch Phone" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchPhone = LogicalKeyboardKey(0x001000c008c, debugName: kReleaseMode ? null : 'Launch Phone');

  /// Represents the logical "Program Guide" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey programGuide = LogicalKeyboardKey(0x001000c008d, debugName: kReleaseMode ? null : 'Program Guide');

  /// Represents the logical "Exit" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey exit = LogicalKeyboardKey(0x001000c0094, debugName: kReleaseMode ? null : 'Exit');

  /// Represents the logical "Channel Up" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey channelUp = LogicalKeyboardKey(0x001000c009c, debugName: kReleaseMode ? null : 'Channel Up');

  /// Represents the logical "Channel Down" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey channelDown = LogicalKeyboardKey(0x001000c009d, debugName: kReleaseMode ? null : 'Channel Down');

  /// Represents the logical "Media Play" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaPlay = LogicalKeyboardKey(0x001000c00b0, debugName: kReleaseMode ? null : 'Media Play');

  /// Represents the logical "Media Record" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaRecord = LogicalKeyboardKey(0x001000c00b2, debugName: kReleaseMode ? null : 'Media Record');

  /// Represents the logical "Media Fast Forward" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaFastForward = LogicalKeyboardKey(0x001000c00b3, debugName: kReleaseMode ? null : 'Media Fast Forward');

  /// Represents the logical "Media Rewind" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaRewind = LogicalKeyboardKey(0x001000c00b4, debugName: kReleaseMode ? null : 'Media Rewind');

  /// Represents the logical "Media Track Next" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaTrackNext = LogicalKeyboardKey(0x001000c00b5, debugName: kReleaseMode ? null : 'Media Track Next');

  /// Represents the logical "Media Track Previous" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaTrackPrevious = LogicalKeyboardKey(0x001000c00b6, debugName: kReleaseMode ? null : 'Media Track Previous');

  /// Represents the logical "Media Stop" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaStop = LogicalKeyboardKey(0x001000c00b7, debugName: kReleaseMode ? null : 'Media Stop');

  /// Represents the logical "Eject" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey eject = LogicalKeyboardKey(0x001000c00b8, debugName: kReleaseMode ? null : 'Eject');

  /// Represents the logical "Media Play Pause" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaPlayPause = LogicalKeyboardKey(0x001000c00cd, debugName: kReleaseMode ? null : 'Media Play Pause');

  /// Represents the logical "Speech Input Toggle" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey speechInputToggle = LogicalKeyboardKey(0x001000c00cf, debugName: kReleaseMode ? null : 'Speech Input Toggle');

  /// Represents the logical "Bass Boost" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey bassBoost = LogicalKeyboardKey(0x001000c00e5, debugName: kReleaseMode ? null : 'Bass Boost');

  /// Represents the logical "Media Select" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mediaSelect = LogicalKeyboardKey(0x001000c0183, debugName: kReleaseMode ? null : 'Media Select');

  /// Represents the logical "Launch Word Processor" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchWordProcessor = LogicalKeyboardKey(0x001000c0184, debugName: kReleaseMode ? null : 'Launch Word Processor');

  /// Represents the logical "Launch Spreadsheet" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchSpreadsheet = LogicalKeyboardKey(0x001000c0186, debugName: kReleaseMode ? null : 'Launch Spreadsheet');

  /// Represents the logical "Launch Mail" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchMail = LogicalKeyboardKey(0x001000c018a, debugName: kReleaseMode ? null : 'Launch Mail');

  /// Represents the logical "Launch Contacts" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchContacts = LogicalKeyboardKey(0x001000c018d, debugName: kReleaseMode ? null : 'Launch Contacts');

  /// Represents the logical "Launch Calendar" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchCalendar = LogicalKeyboardKey(0x001000c018e, debugName: kReleaseMode ? null : 'Launch Calendar');

  /// Represents the logical "Launch App2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchApp2 = LogicalKeyboardKey(0x001000c0192, debugName: kReleaseMode ? null : 'Launch App2');

  /// Represents the logical "Launch App1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchApp1 = LogicalKeyboardKey(0x001000c0194, debugName: kReleaseMode ? null : 'Launch App1');

  /// Represents the logical "Launch Internet Browser" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchInternetBrowser = LogicalKeyboardKey(0x001000c0196, debugName: kReleaseMode ? null : 'Launch Internet Browser');

  /// Represents the logical "Log Off" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey logOff = LogicalKeyboardKey(0x001000c019c, debugName: kReleaseMode ? null : 'Log Off');

  /// Represents the logical "Lock Screen" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey lockScreen = LogicalKeyboardKey(0x001000c019e, debugName: kReleaseMode ? null : 'Lock Screen');

  /// Represents the logical "Launch Control Panel" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchControlPanel = LogicalKeyboardKey(0x001000c019f, debugName: kReleaseMode ? null : 'Launch Control Panel');

  /// Represents the logical "Select Task" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey selectTask = LogicalKeyboardKey(0x001000c01a2, debugName: kReleaseMode ? null : 'Select Task');

  /// Represents the logical "Launch Documents" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchDocuments = LogicalKeyboardKey(0x001000c01a7, debugName: kReleaseMode ? null : 'Launch Documents');

  /// Represents the logical "Spell Check" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey spellCheck = LogicalKeyboardKey(0x001000c01ab, debugName: kReleaseMode ? null : 'Spell Check');

  /// Represents the logical "Launch Keyboard Layout" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchKeyboardLayout = LogicalKeyboardKey(0x001000c01ae, debugName: kReleaseMode ? null : 'Launch Keyboard Layout');

  /// Represents the logical "Launch Screen Saver" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchScreenSaver = LogicalKeyboardKey(0x001000c01b1, debugName: kReleaseMode ? null : 'Launch Screen Saver');

  /// Represents the logical "Launch Assistant" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchAssistant = LogicalKeyboardKey(0x001000c01cb, debugName: kReleaseMode ? null : 'Launch Assistant');

  /// Represents the logical "Launch Audio Browser" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey launchAudioBrowser = LogicalKeyboardKey(0x001000c01b7, debugName: kReleaseMode ? null : 'Launch Audio Browser');

  /// Represents the logical "New Key" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey newKey = LogicalKeyboardKey(0x001000c0201, debugName: kReleaseMode ? null : 'New Key');

  /// Represents the logical "Close" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey close = LogicalKeyboardKey(0x001000c0203, debugName: kReleaseMode ? null : 'Close');

  /// Represents the logical "Save" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey save = LogicalKeyboardKey(0x001000c0207, debugName: kReleaseMode ? null : 'Save');

  /// Represents the logical "Print" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey print = LogicalKeyboardKey(0x001000c0208, debugName: kReleaseMode ? null : 'Print');

  /// Represents the logical "Browser Search" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey browserSearch = LogicalKeyboardKey(0x001000c0221, debugName: kReleaseMode ? null : 'Browser Search');

  /// Represents the logical "Browser Home" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey browserHome = LogicalKeyboardKey(0x001000c0223, debugName: kReleaseMode ? null : 'Browser Home');

  /// Represents the logical "Browser Back" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey browserBack = LogicalKeyboardKey(0x001000c0224, debugName: kReleaseMode ? null : 'Browser Back');

  /// Represents the logical "Browser Forward" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey browserForward = LogicalKeyboardKey(0x001000c0225, debugName: kReleaseMode ? null : 'Browser Forward');

  /// Represents the logical "Browser Stop" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey browserStop = LogicalKeyboardKey(0x001000c0226, debugName: kReleaseMode ? null : 'Browser Stop');

  /// Represents the logical "Browser Refresh" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey browserRefresh = LogicalKeyboardKey(0x001000c0227, debugName: kReleaseMode ? null : 'Browser Refresh');

  /// Represents the logical "Browser Favorites" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey browserFavorites = LogicalKeyboardKey(0x001000c022a, debugName: kReleaseMode ? null : 'Browser Favorites');

  /// Represents the logical "Zoom In" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey zoomIn = LogicalKeyboardKey(0x001000c022d, debugName: kReleaseMode ? null : 'Zoom In');

  /// Represents the logical "Zoom Out" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey zoomOut = LogicalKeyboardKey(0x001000c022e, debugName: kReleaseMode ? null : 'Zoom Out');

  /// Represents the logical "Zoom Toggle" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey zoomToggle = LogicalKeyboardKey(0x001000c0232, debugName: kReleaseMode ? null : 'Zoom Toggle');

  /// Represents the logical "Redo" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey redo = LogicalKeyboardKey(0x001000c0279, debugName: kReleaseMode ? null : 'Redo');

  /// Represents the logical "Mail Reply" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mailReply = LogicalKeyboardKey(0x001000c0289, debugName: kReleaseMode ? null : 'Mail Reply');

  /// Represents the logical "Mail Forward" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mailForward = LogicalKeyboardKey(0x001000c028b, debugName: kReleaseMode ? null : 'Mail Forward');

  /// Represents the logical "Mail Send" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey mailSend = LogicalKeyboardKey(0x001000c028c, debugName: kReleaseMode ? null : 'Mail Send');

  /// Represents the logical "Keyboard Layout Select" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey keyboardLayoutSelect = LogicalKeyboardKey(0x001000c029d, debugName: kReleaseMode ? null : 'Keyboard Layout Select');

  /// Represents the logical "Show All Windows" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey showAllWindows = LogicalKeyboardKey(0x001000c029f, debugName: kReleaseMode ? null : 'Show All Windows');

  /// Represents the logical "Game Button 1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton1 = LogicalKeyboardKey(0x0010005ff01, debugName: kReleaseMode ? null : 'Game Button 1');

  /// Represents the logical "Game Button 2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton2 = LogicalKeyboardKey(0x0010005ff02, debugName: kReleaseMode ? null : 'Game Button 2');

  /// Represents the logical "Game Button 3" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton3 = LogicalKeyboardKey(0x0010005ff03, debugName: kReleaseMode ? null : 'Game Button 3');

  /// Represents the logical "Game Button 4" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton4 = LogicalKeyboardKey(0x0010005ff04, debugName: kReleaseMode ? null : 'Game Button 4');

  /// Represents the logical "Game Button 5" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton5 = LogicalKeyboardKey(0x0010005ff05, debugName: kReleaseMode ? null : 'Game Button 5');

  /// Represents the logical "Game Button 6" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton6 = LogicalKeyboardKey(0x0010005ff06, debugName: kReleaseMode ? null : 'Game Button 6');

  /// Represents the logical "Game Button 7" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton7 = LogicalKeyboardKey(0x0010005ff07, debugName: kReleaseMode ? null : 'Game Button 7');

  /// Represents the logical "Game Button 8" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton8 = LogicalKeyboardKey(0x0010005ff08, debugName: kReleaseMode ? null : 'Game Button 8');

  /// Represents the logical "Game Button 9" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton9 = LogicalKeyboardKey(0x0010005ff09, debugName: kReleaseMode ? null : 'Game Button 9');

  /// Represents the logical "Game Button 10" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton10 = LogicalKeyboardKey(0x0010005ff0a, debugName: kReleaseMode ? null : 'Game Button 10');

  /// Represents the logical "Game Button 11" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton11 = LogicalKeyboardKey(0x0010005ff0b, debugName: kReleaseMode ? null : 'Game Button 11');

  /// Represents the logical "Game Button 12" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton12 = LogicalKeyboardKey(0x0010005ff0c, debugName: kReleaseMode ? null : 'Game Button 12');

  /// Represents the logical "Game Button 13" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton13 = LogicalKeyboardKey(0x0010005ff0d, debugName: kReleaseMode ? null : 'Game Button 13');

  /// Represents the logical "Game Button 14" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton14 = LogicalKeyboardKey(0x0010005ff0e, debugName: kReleaseMode ? null : 'Game Button 14');

  /// Represents the logical "Game Button 15" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton15 = LogicalKeyboardKey(0x0010005ff0f, debugName: kReleaseMode ? null : 'Game Button 15');

  /// Represents the logical "Game Button 16" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButton16 = LogicalKeyboardKey(0x0010005ff10, debugName: kReleaseMode ? null : 'Game Button 16');

  /// Represents the logical "Game Button A" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonA = LogicalKeyboardKey(0x0010005ff11, debugName: kReleaseMode ? null : 'Game Button A');

  /// Represents the logical "Game Button B" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonB = LogicalKeyboardKey(0x0010005ff12, debugName: kReleaseMode ? null : 'Game Button B');

  /// Represents the logical "Game Button C" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonC = LogicalKeyboardKey(0x0010005ff13, debugName: kReleaseMode ? null : 'Game Button C');

  /// Represents the logical "Game Button Left 1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonLeft1 = LogicalKeyboardKey(0x0010005ff14, debugName: kReleaseMode ? null : 'Game Button Left 1');

  /// Represents the logical "Game Button Left 2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonLeft2 = LogicalKeyboardKey(0x0010005ff15, debugName: kReleaseMode ? null : 'Game Button Left 2');

  /// Represents the logical "Game Button Mode" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonMode = LogicalKeyboardKey(0x0010005ff16, debugName: kReleaseMode ? null : 'Game Button Mode');

  /// Represents the logical "Game Button Right 1" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonRight1 = LogicalKeyboardKey(0x0010005ff17, debugName: kReleaseMode ? null : 'Game Button Right 1');

  /// Represents the logical "Game Button Right 2" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonRight2 = LogicalKeyboardKey(0x0010005ff18, debugName: kReleaseMode ? null : 'Game Button Right 2');

  /// Represents the logical "Game Button Select" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonSelect = LogicalKeyboardKey(0x0010005ff19, debugName: kReleaseMode ? null : 'Game Button Select');

  /// Represents the logical "Game Button Start" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonStart = LogicalKeyboardKey(0x0010005ff1a, debugName: kReleaseMode ? null : 'Game Button Start');

  /// Represents the logical "Game Button Thumb Left" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonThumbLeft = LogicalKeyboardKey(0x0010005ff1b, debugName: kReleaseMode ? null : 'Game Button Thumb Left');

  /// Represents the logical "Game Button Thumb Right" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonThumbRight = LogicalKeyboardKey(0x0010005ff1c, debugName: kReleaseMode ? null : 'Game Button Thumb Right');

  /// Represents the logical "Game Button X" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonX = LogicalKeyboardKey(0x0010005ff1d, debugName: kReleaseMode ? null : 'Game Button X');

  /// Represents the logical "Game Button Y" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonY = LogicalKeyboardKey(0x0010005ff1e, debugName: kReleaseMode ? null : 'Game Button Y');

  /// Represents the logical "Game Button Z" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey gameButtonZ = LogicalKeyboardKey(0x0010005ff1f, debugName: kReleaseMode ? null : 'Game Button Z');

  /// Represents the logical "Fn" key on the keyboard.
  ///
  /// See the function [RawKeyEvent.logicalKey] for more information.
  static const LogicalKeyboardKey fn = LogicalKeyboardKey(0x00100000012, debugName: kReleaseMode ? null : 'Fn');

  /// Represents the logical "Shift" key on the keyboard.
  ///
  /// This key represents the union of the keys {shiftLeft, shiftRight} when
  /// comparing keys. This key will never be generated directly, its main use is
  /// in defining key maps.
  static const LogicalKeyboardKey shift = LogicalKeyboardKey(0x201000700e1, debugName: kReleaseMode ? null : 'Shift');

  /// Represents the logical "Meta" key on the keyboard.
  ///
  /// This key represents the union of the keys {metaLeft, metaRight} when
  /// comparing keys. This key will never be generated directly, its main use is
  /// in defining key maps.
  static const LogicalKeyboardKey meta = LogicalKeyboardKey(0x201000700e3, debugName: kReleaseMode ? null : 'Meta');

  /// Represents the logical "Alt" key on the keyboard.
  ///
  /// This key represents the union of the keys {altLeft, altRight} when
  /// comparing keys. This key will never be generated directly, its main use is
  /// in defining key maps.
  static const LogicalKeyboardKey alt = LogicalKeyboardKey(0x201000700e2, debugName: kReleaseMode ? null : 'Alt');

  /// Represents the logical "Control" key on the keyboard.
  ///
  /// This key represents the union of the keys {controlLeft, controlRight} when
  /// comparing keys. This key will never be generated directly, its main use is
  /// in defining key maps.
  static const LogicalKeyboardKey control = LogicalKeyboardKey(0x201000700e0, debugName: kReleaseMode ? null : 'Control');

  // A list of all predefined constant LogicalKeyboardKeys so they can be
  // searched.
  static const Map<int, LogicalKeyboardKey> _knownLogicalKeys = <int, LogicalKeyboardKey>{
    0x0100000000: none,
    0x0100000010: hyper,
    0x0100000011: superKey,
    0x0100000013: fnLock,
    0x0100000014: suspend,
    0x0100000015: resume,
    0x0100000016: turbo,
    0x0100010082: sleep,
    0x0100010083: wakeUp,
    0x01000100b5: displayToggleIntExt,
    0x0100070000: usbReserved,
    0x0100070001: usbErrorRollOver,
    0x0100070002: usbPostFail,
    0x0100070003: usbErrorUndefined,
    0x0000000061: keyA,
    0x0000000062: keyB,
    0x0000000063: keyC,
    0x0000000064: keyD,
    0x0000000065: keyE,
    0x0000000066: keyF,
    0x0000000067: keyG,
    0x0000000068: keyH,
    0x0000000069: keyI,
    0x000000006a: keyJ,
    0x000000006b: keyK,
    0x000000006c: keyL,
    0x000000006d: keyM,
    0x000000006e: keyN,
    0x000000006f: keyO,
    0x0000000070: keyP,
    0x0000000071: keyQ,
    0x0000000072: keyR,
    0x0000000073: keyS,
    0x0000000074: keyT,
    0x0000000075: keyU,
    0x0000000076: keyV,
    0x0000000077: keyW,
    0x0000000078: keyX,
    0x0000000079: keyY,
    0x000000007a: keyZ,
    0x0000000031: digit1,
    0x0000000032: digit2,
    0x0000000033: digit3,
    0x0000000034: digit4,
    0x0000000035: digit5,
    0x0000000036: digit6,
    0x0000000037: digit7,
    0x0000000038: digit8,
    0x0000000039: digit9,
    0x0000000030: digit0,
    0x0100070028: enter,
    0x0100070029: escape,
    0x010007002a: backspace,
    0x010007002b: tab,
    0x0000000020: space,
    0x000000002d: minus,
    0x000000003d: equal,
    0x000000005b: bracketLeft,
    0x000000005d: bracketRight,
    0x000000005c: backslash,
    0x000000003b: semicolon,
    0x0000000027: quote,
    0x0000000060: backquote,
    0x000000002c: comma,
    0x000000002e: period,
    0x000000002f: slash,
    0x0100070039: capsLock,
    0x010007003a: f1,
    0x010007003b: f2,
    0x010007003c: f3,
    0x010007003d: f4,
    0x010007003e: f5,
    0x010007003f: f6,
    0x0100070040: f7,
    0x0100070041: f8,
    0x0100070042: f9,
    0x0100070043: f10,
    0x0100070044: f11,
    0x0100070045: f12,
    0x0100070046: printScreen,
    0x0100070047: scrollLock,
    0x0100070048: pause,
    0x0100070049: insert,
    0x010007004a: home,
    0x010007004b: pageUp,
    0x010007004c: delete,
    0x010007004d: end,
    0x010007004e: pageDown,
    0x010007004f: arrowRight,
    0x0100070050: arrowLeft,
    0x0100070051: arrowDown,
    0x0100070052: arrowUp,
    0x0100070053: numLock,
    0x0100070054: numpadDivide,
    0x0100070055: numpadMultiply,
    0x0100070056: numpadSubtract,
    0x0100070057: numpadAdd,
    0x0100070058: numpadEnter,
    0x0100070059: numpad1,
    0x010007005a: numpad2,
    0x010007005b: numpad3,
    0x010007005c: numpad4,
    0x010007005d: numpad5,
    0x010007005e: numpad6,
    0x010007005f: numpad7,
    0x0100070060: numpad8,
    0x0100070061: numpad9,
    0x0100070062: numpad0,
    0x0100070063: numpadDecimal,
    0x0100070064: intlBackslash,
    0x0100070065: contextMenu,
    0x0100070066: power,
    0x0100070067: numpadEqual,
    0x0100070068: f13,
    0x0100070069: f14,
    0x010007006a: f15,
    0x010007006b: f16,
    0x010007006c: f17,
    0x010007006d: f18,
    0x010007006e: f19,
    0x010007006f: f20,
    0x0100070070: f21,
    0x0100070071: f22,
    0x0100070072: f23,
    0x0100070073: f24,
    0x0100070074: open,
    0x0100070075: help,
    0x0100070077: select,
    0x0100070079: again,
    0x010007007a: undo,
    0x010007007b: cut,
    0x010007007c: copy,
    0x010007007d: paste,
    0x010007007e: find,
    0x010007007f: audioVolumeMute,
    0x0100070080: audioVolumeUp,
    0x0100070081: audioVolumeDown,
    0x0100070085: numpadComma,
    0x0100070087: intlRo,
    0x0100070088: kanaMode,
    0x0100070089: intlYen,
    0x010007008a: convert,
    0x010007008b: nonConvert,
    0x0100070090: lang1,
    0x0100070091: lang2,
    0x0100070092: lang3,
    0x0100070093: lang4,
    0x0100070094: lang5,
    0x010007009b: abort,
    0x01000700a3: props,
    0x01000700b6: numpadParenLeft,
    0x01000700b7: numpadParenRight,
    0x01000700bb: numpadBackspace,
    0x01000700d0: numpadMemoryStore,
    0x01000700d1: numpadMemoryRecall,
    0x01000700d2: numpadMemoryClear,
    0x01000700d3: numpadMemoryAdd,
    0x01000700d4: numpadMemorySubtract,
    0x01000700d7: numpadSignChange,
    0x01000700d8: numpadClear,
    0x01000700d9: numpadClearEntry,
    0x01000700e0: controlLeft,
    0x01000700e1: shiftLeft,
    0x01000700e2: altLeft,
    0x01000700e3: metaLeft,
    0x01000700e4: controlRight,
    0x01000700e5: shiftRight,
    0x01000700e6: altRight,
    0x01000700e7: metaRight,
    0x01000c0060: info,
    0x01000c0061: closedCaptionToggle,
    0x01000c006f: brightnessUp,
    0x01000c0070: brightnessDown,
    0x01000c0072: brightnessToggle,
    0x01000c0073: brightnessMinimum,
    0x01000c0074: brightnessMaximum,
    0x01000c0075: brightnessAuto,
    0x01000c0083: mediaLast,
    0x01000c008c: launchPhone,
    0x01000c008d: programGuide,
    0x01000c0094: exit,
    0x01000c009c: channelUp,
    0x01000c009d: channelDown,
    0x01000c00b0: mediaPlay,
    0x01000c00b2: mediaRecord,
    0x01000c00b3: mediaFastForward,
    0x01000c00b4: mediaRewind,
    0x01000c00b5: mediaTrackNext,
    0x01000c00b6: mediaTrackPrevious,
    0x01000c00b7: mediaStop,
    0x01000c00b8: eject,
    0x01000c00cd: mediaPlayPause,
    0x01000c00cf: speechInputToggle,
    0x01000c00e5: bassBoost,
    0x01000c0183: mediaSelect,
    0x01000c0184: launchWordProcessor,
    0x01000c0186: launchSpreadsheet,
    0x01000c018a: launchMail,
    0x01000c018d: launchContacts,
    0x01000c018e: launchCalendar,
    0x01000c0192: launchApp2,
    0x01000c0194: launchApp1,
    0x01000c0196: launchInternetBrowser,
    0x01000c019c: logOff,
    0x01000c019e: lockScreen,
    0x01000c019f: launchControlPanel,
    0x01000c01a2: selectTask,
    0x01000c01a7: launchDocuments,
    0x01000c01ab: spellCheck,
    0x01000c01ae: launchKeyboardLayout,
    0x01000c01b1: launchScreenSaver,
    0x01000c01cb: launchAssistant,
    0x01000c01b7: launchAudioBrowser,
    0x01000c0201: newKey,
    0x01000c0203: close,
    0x01000c0207: save,
    0x01000c0208: print,
    0x01000c0221: browserSearch,
    0x01000c0223: browserHome,
    0x01000c0224: browserBack,
    0x01000c0225: browserForward,
    0x01000c0226: browserStop,
    0x01000c0227: browserRefresh,
    0x01000c022a: browserFavorites,
    0x01000c022d: zoomIn,
    0x01000c022e: zoomOut,
    0x01000c0232: zoomToggle,
    0x01000c0279: redo,
    0x01000c0289: mailReply,
    0x01000c028b: mailForward,
    0x01000c028c: mailSend,
    0x01000c029d: keyboardLayoutSelect,
    0x01000c029f: showAllWindows,
    0x010005ff01: gameButton1,
    0x010005ff02: gameButton2,
    0x010005ff03: gameButton3,
    0x010005ff04: gameButton4,
    0x010005ff05: gameButton5,
    0x010005ff06: gameButton6,
    0x010005ff07: gameButton7,
    0x010005ff08: gameButton8,
    0x010005ff09: gameButton9,
    0x010005ff0a: gameButton10,
    0x010005ff0b: gameButton11,
    0x010005ff0c: gameButton12,
    0x010005ff0d: gameButton13,
    0x010005ff0e: gameButton14,
    0x010005ff0f: gameButton15,
    0x010005ff10: gameButton16,
    0x010005ff11: gameButtonA,
    0x010005ff12: gameButtonB,
    0x010005ff13: gameButtonC,
    0x010005ff14: gameButtonLeft1,
    0x010005ff15: gameButtonLeft2,
    0x010005ff16: gameButtonMode,
    0x010005ff17: gameButtonRight1,
    0x010005ff18: gameButtonRight2,
    0x010005ff19: gameButtonSelect,
    0x010005ff1a: gameButtonStart,
    0x010005ff1b: gameButtonThumbLeft,
    0x010005ff1c: gameButtonThumbRight,
    0x010005ff1d: gameButtonX,
    0x010005ff1e: gameButtonY,
    0x010005ff1f: gameButtonZ,
    0x0100000012: fn,
    0x201000700e1: shift,
    0x201000700e3: meta,
    0x201000700e2: alt,
    0x201000700e0: control,
  };

  // A map of keys to the pseudo-key synonym for that key. Used by getSynonyms.
  static final Map<LogicalKeyboardKey, LogicalKeyboardKey> _synonyms = <LogicalKeyboardKey, LogicalKeyboardKey>{
    shiftLeft: shift,
    shiftRight: shift,
    metaLeft: meta,
    metaRight: meta,
    altLeft: alt,
    altRight: alt,
    controlLeft: control,
    controlRight: control,
  };
}

/// A class with static values that describe the keys that are returned from
/// [RawKeyEvent.physicalKey].
///
/// These represent *physical* keys, which are keys which represent a particular
/// key location on a QWERTY keyboard. It ignores any modifiers, modes, or
/// keyboard layouts which may be in effect. This is contrast to
/// [LogicalKeyboardKey], which represents a logical key interpreted in the
/// context of modifiers, modes, and/or keyboard layouts.
///
/// As an example, if you wanted a game where the key next to the CAPS LOCK (the
/// "A" key on a QWERTY keyboard) moved the player to the left, you'd want to
/// look at the physical key to make sure that regardless of the character the
/// key produces, you got the key that is in that location on the keyboard.
///
/// Conversely, if you wanted to implement an app where the "Q" key "quit"
/// something, you'd want to look at the logical key to detect this, since you
/// would like to have it match the key with "Q" on it, instead of always
/// looking for "the key next next to the TAB key", since on a French keyboard,
/// the key next to the TAB key has an "A" on it.
///
/// {@tool sample --template=stateful_widget_scaffold}
/// This example shows how to detect if the user has selected the physical key
/// to the right of the CAPS LOCK key.
///
/// ```dart imports
/// import 'package:flutter/services.dart';
/// ```
///
/// ```dart
/// // The node used to request the keyboard focus.
/// final FocusNode _focusNode = FocusNode();
/// // The message to display.
/// String _message;
///
/// // Focus nodes need to be disposed.
/// @override
/// void dispose() {
///   _focusNode.dispose();
///   super.dispose();
/// }
///
/// // Handles the key events from the RawKeyboardListener and update the
/// // _message.
/// void _handleKeyEvent(RawKeyEvent event) {
///   setState(() {
///     if (event.physicalKey == PhysicalKeyboardKey.keyA) {
///       _message = 'Pressed the key next to CAPS LOCK!';
///     } else {
///       _message = 'Wrong key.';
///     }
///   });
/// }
///
/// @override
/// Widget build(BuildContext context) {
///   final TextTheme textTheme = Theme.of(context).textTheme;
///   return Container(
///     color: Colors.white,
///     alignment: Alignment.center,
///     child: DefaultTextStyle(
///       style: textTheme.display1,
///       child: RawKeyboardListener(
///         focusNode: _focusNode,
///         onKey: _handleKeyEvent,
///         child: AnimatedBuilder(
///           animation: _focusNode,
///           builder: (BuildContext context, Widget child) {
///             if (!_focusNode.hasFocus) {
///               return GestureDetector(
///                 onTap: () {
///                   FocusScope.of(context).requestFocus(_focusNode);
///                 },
///                 child: Text('Tap to focus'),
///               );
///             }
///             return Text(_message ?? 'Press a key');
///           },
///         ),
///       ),
///     ),
///   );
/// }
/// ```
/// {@end-tool}
///
/// See also:
///
///  * [RawKeyEvent], the keyboard event object received by widgets that listen
///    to keyboard events.
///  * [RawKeyboardListener], a widget used to listen to and supply handlers for
///    keyboard events.
class PhysicalKeyboardKey extends KeyboardKey {
  /// Creates a PhysicalKeyboardKey object with an optional debug name.
  ///
  /// The [usbHidUsage] must not be null.
  ///
  /// {@tool snippet}
  /// To save executable size, it is recommended that the [debugName] be null in
  /// release mode. You can do this using the [kReleaseMode] constant.
  ///
  /// ```dart
  /// const PhysicalKeyboardKey(0x0000ffff, debugName: kReleaseMode ? null : 'Special Key')
  /// ```
  /// {@end-tool}
  const PhysicalKeyboardKey(this.usbHidUsage, {this.debugName})
      : assert(usbHidUsage != null);

  /// The unique USB HID usage ID of this physical key on the keyboard.
  ///
  /// Due to the variations in platform APIs, this may not be the actual HID
  /// usage code from the hardware, but a value derived from available
  /// information on the platform.
  ///
  /// See <https://www.usb.org/sites/default/files/documents/hut1_12v2.pdf>
  /// for the HID usage values and their meanings.
  final int usbHidUsage;

  /// The debug string to print for this keyboard key, which will be null in
  /// release mode.
  final String debugName;

  /// Finds a known [PhysicalKeyboardKey] that matches the given USB HID usage
  /// code.
  static PhysicalKeyboardKey findKeyByCode(int usageCode) => _knownPhysicalKeys[usageCode];

  @override
  int get hashCode => usbHidUsage.hashCode;

  @override
  bool operator ==(Object other) {
    if (other.runtimeType != runtimeType) {
      return false;
    }
    return other is PhysicalKeyboardKey
        && other.usbHidUsage == usbHidUsage;
  }

  @override
  void debugFillProperties(DiagnosticPropertiesBuilder properties) {
    super.debugFillProperties(properties);
    properties.add(StringProperty('usbHidUsage', '0x${usbHidUsage.toRadixString(16).padLeft(8, '0')}', showName: true));
    properties.add(StringProperty('debugName', debugName, showName: true, defaultValue: null));
  }

  // Key constants for all keyboard keys in the USB HID specification at the
  // time Flutter was built.

  /// Represents the location of the "None" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey none = PhysicalKeyboardKey(0x00000000, debugName: kReleaseMode ? null : 'None');

  /// Represents the location of the "Hyper" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey hyper = PhysicalKeyboardKey(0x00000010, debugName: kReleaseMode ? null : 'Hyper');

  /// Represents the location of the "Super Key" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey superKey = PhysicalKeyboardKey(0x00000011, debugName: kReleaseMode ? null : 'Super Key');

  /// Represents the location of the "Fn Lock" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey fnLock = PhysicalKeyboardKey(0x00000013, debugName: kReleaseMode ? null : 'Fn Lock');

  /// Represents the location of the "Suspend" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey suspend = PhysicalKeyboardKey(0x00000014, debugName: kReleaseMode ? null : 'Suspend');

  /// Represents the location of the "Resume" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey resume = PhysicalKeyboardKey(0x00000015, debugName: kReleaseMode ? null : 'Resume');

  /// Represents the location of the "Turbo" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey turbo = PhysicalKeyboardKey(0x00000016, debugName: kReleaseMode ? null : 'Turbo');

  /// Represents the location of the "Sleep" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey sleep = PhysicalKeyboardKey(0x00010082, debugName: kReleaseMode ? null : 'Sleep');

  /// Represents the location of the "Wake Up" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey wakeUp = PhysicalKeyboardKey(0x00010083, debugName: kReleaseMode ? null : 'Wake Up');

  /// Represents the location of the "Display Toggle Int Ext" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey displayToggleIntExt = PhysicalKeyboardKey(0x000100b5, debugName: kReleaseMode ? null : 'Display Toggle Int Ext');

  /// Represents the location of the "Usb Reserved" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey usbReserved = PhysicalKeyboardKey(0x00070000, debugName: kReleaseMode ? null : 'Usb Reserved');

  /// Represents the location of the "Usb Error Roll Over" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey usbErrorRollOver = PhysicalKeyboardKey(0x00070001, debugName: kReleaseMode ? null : 'Usb Error Roll Over');

  /// Represents the location of the "Usb Post Fail" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey usbPostFail = PhysicalKeyboardKey(0x00070002, debugName: kReleaseMode ? null : 'Usb Post Fail');

  /// Represents the location of the "Usb Error Undefined" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey usbErrorUndefined = PhysicalKeyboardKey(0x00070003, debugName: kReleaseMode ? null : 'Usb Error Undefined');

  /// Represents the location of the "Key A" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyA = PhysicalKeyboardKey(0x00070004, debugName: kReleaseMode ? null : 'Key A');

  /// Represents the location of the "Key B" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyB = PhysicalKeyboardKey(0x00070005, debugName: kReleaseMode ? null : 'Key B');

  /// Represents the location of the "Key C" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyC = PhysicalKeyboardKey(0x00070006, debugName: kReleaseMode ? null : 'Key C');

  /// Represents the location of the "Key D" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyD = PhysicalKeyboardKey(0x00070007, debugName: kReleaseMode ? null : 'Key D');

  /// Represents the location of the "Key E" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyE = PhysicalKeyboardKey(0x00070008, debugName: kReleaseMode ? null : 'Key E');

  /// Represents the location of the "Key F" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyF = PhysicalKeyboardKey(0x00070009, debugName: kReleaseMode ? null : 'Key F');

  /// Represents the location of the "Key G" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyG = PhysicalKeyboardKey(0x0007000a, debugName: kReleaseMode ? null : 'Key G');

  /// Represents the location of the "Key H" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyH = PhysicalKeyboardKey(0x0007000b, debugName: kReleaseMode ? null : 'Key H');

  /// Represents the location of the "Key I" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyI = PhysicalKeyboardKey(0x0007000c, debugName: kReleaseMode ? null : 'Key I');

  /// Represents the location of the "Key J" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyJ = PhysicalKeyboardKey(0x0007000d, debugName: kReleaseMode ? null : 'Key J');

  /// Represents the location of the "Key K" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyK = PhysicalKeyboardKey(0x0007000e, debugName: kReleaseMode ? null : 'Key K');

  /// Represents the location of the "Key L" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyL = PhysicalKeyboardKey(0x0007000f, debugName: kReleaseMode ? null : 'Key L');

  /// Represents the location of the "Key M" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyM = PhysicalKeyboardKey(0x00070010, debugName: kReleaseMode ? null : 'Key M');

  /// Represents the location of the "Key N" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyN = PhysicalKeyboardKey(0x00070011, debugName: kReleaseMode ? null : 'Key N');

  /// Represents the location of the "Key O" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyO = PhysicalKeyboardKey(0x00070012, debugName: kReleaseMode ? null : 'Key O');

  /// Represents the location of the "Key P" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyP = PhysicalKeyboardKey(0x00070013, debugName: kReleaseMode ? null : 'Key P');

  /// Represents the location of the "Key Q" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyQ = PhysicalKeyboardKey(0x00070014, debugName: kReleaseMode ? null : 'Key Q');

  /// Represents the location of the "Key R" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyR = PhysicalKeyboardKey(0x00070015, debugName: kReleaseMode ? null : 'Key R');

  /// Represents the location of the "Key S" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyS = PhysicalKeyboardKey(0x00070016, debugName: kReleaseMode ? null : 'Key S');

  /// Represents the location of the "Key T" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyT = PhysicalKeyboardKey(0x00070017, debugName: kReleaseMode ? null : 'Key T');

  /// Represents the location of the "Key U" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyU = PhysicalKeyboardKey(0x00070018, debugName: kReleaseMode ? null : 'Key U');

  /// Represents the location of the "Key V" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyV = PhysicalKeyboardKey(0x00070019, debugName: kReleaseMode ? null : 'Key V');

  /// Represents the location of the "Key W" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyW = PhysicalKeyboardKey(0x0007001a, debugName: kReleaseMode ? null : 'Key W');

  /// Represents the location of the "Key X" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyX = PhysicalKeyboardKey(0x0007001b, debugName: kReleaseMode ? null : 'Key X');

  /// Represents the location of the "Key Y" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyY = PhysicalKeyboardKey(0x0007001c, debugName: kReleaseMode ? null : 'Key Y');

  /// Represents the location of the "Key Z" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyZ = PhysicalKeyboardKey(0x0007001d, debugName: kReleaseMode ? null : 'Key Z');

  /// Represents the location of the "Digit 1" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit1 = PhysicalKeyboardKey(0x0007001e, debugName: kReleaseMode ? null : 'Digit 1');

  /// Represents the location of the "Digit 2" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit2 = PhysicalKeyboardKey(0x0007001f, debugName: kReleaseMode ? null : 'Digit 2');

  /// Represents the location of the "Digit 3" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit3 = PhysicalKeyboardKey(0x00070020, debugName: kReleaseMode ? null : 'Digit 3');

  /// Represents the location of the "Digit 4" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit4 = PhysicalKeyboardKey(0x00070021, debugName: kReleaseMode ? null : 'Digit 4');

  /// Represents the location of the "Digit 5" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit5 = PhysicalKeyboardKey(0x00070022, debugName: kReleaseMode ? null : 'Digit 5');

  /// Represents the location of the "Digit 6" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit6 = PhysicalKeyboardKey(0x00070023, debugName: kReleaseMode ? null : 'Digit 6');

  /// Represents the location of the "Digit 7" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit7 = PhysicalKeyboardKey(0x00070024, debugName: kReleaseMode ? null : 'Digit 7');

  /// Represents the location of the "Digit 8" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit8 = PhysicalKeyboardKey(0x00070025, debugName: kReleaseMode ? null : 'Digit 8');

  /// Represents the location of the "Digit 9" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit9 = PhysicalKeyboardKey(0x00070026, debugName: kReleaseMode ? null : 'Digit 9');

  /// Represents the location of the "Digit 0" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey digit0 = PhysicalKeyboardKey(0x00070027, debugName: kReleaseMode ? null : 'Digit 0');

  /// Represents the location of the "Enter" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey enter = PhysicalKeyboardKey(0x00070028, debugName: kReleaseMode ? null : 'Enter');

  /// Represents the location of the "Escape" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey escape = PhysicalKeyboardKey(0x00070029, debugName: kReleaseMode ? null : 'Escape');

  /// Represents the location of the "Backspace" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey backspace = PhysicalKeyboardKey(0x0007002a, debugName: kReleaseMode ? null : 'Backspace');

  /// Represents the location of the "Tab" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey tab = PhysicalKeyboardKey(0x0007002b, debugName: kReleaseMode ? null : 'Tab');

  /// Represents the location of the "Space" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey space = PhysicalKeyboardKey(0x0007002c, debugName: kReleaseMode ? null : 'Space');

  /// Represents the location of the "Minus" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey minus = PhysicalKeyboardKey(0x0007002d, debugName: kReleaseMode ? null : 'Minus');

  /// Represents the location of the "Equal" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey equal = PhysicalKeyboardKey(0x0007002e, debugName: kReleaseMode ? null : 'Equal');

  /// Represents the location of the "Bracket Left" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey bracketLeft = PhysicalKeyboardKey(0x0007002f, debugName: kReleaseMode ? null : 'Bracket Left');

  /// Represents the location of the "Bracket Right" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey bracketRight = PhysicalKeyboardKey(0x00070030, debugName: kReleaseMode ? null : 'Bracket Right');

  /// Represents the location of the "Backslash" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey backslash = PhysicalKeyboardKey(0x00070031, debugName: kReleaseMode ? null : 'Backslash');

  /// Represents the location of the "Semicolon" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey semicolon = PhysicalKeyboardKey(0x00070033, debugName: kReleaseMode ? null : 'Semicolon');

  /// Represents the location of the "Quote" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey quote = PhysicalKeyboardKey(0x00070034, debugName: kReleaseMode ? null : 'Quote');

  /// Represents the location of the "Backquote" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey backquote = PhysicalKeyboardKey(0x00070035, debugName: kReleaseMode ? null : 'Backquote');

  /// Represents the location of the "Comma" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey comma = PhysicalKeyboardKey(0x00070036, debugName: kReleaseMode ? null : 'Comma');

  /// Represents the location of the "Period" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey period = PhysicalKeyboardKey(0x00070037, debugName: kReleaseMode ? null : 'Period');

  /// Represents the location of the "Slash" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey slash = PhysicalKeyboardKey(0x00070038, debugName: kReleaseMode ? null : 'Slash');

  /// Represents the location of the "Caps Lock" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey capsLock = PhysicalKeyboardKey(0x00070039, debugName: kReleaseMode ? null : 'Caps Lock');

  /// Represents the location of the "F1" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f1 = PhysicalKeyboardKey(0x0007003a, debugName: kReleaseMode ? null : 'F1');

  /// Represents the location of the "F2" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f2 = PhysicalKeyboardKey(0x0007003b, debugName: kReleaseMode ? null : 'F2');

  /// Represents the location of the "F3" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f3 = PhysicalKeyboardKey(0x0007003c, debugName: kReleaseMode ? null : 'F3');

  /// Represents the location of the "F4" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f4 = PhysicalKeyboardKey(0x0007003d, debugName: kReleaseMode ? null : 'F4');

  /// Represents the location of the "F5" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f5 = PhysicalKeyboardKey(0x0007003e, debugName: kReleaseMode ? null : 'F5');

  /// Represents the location of the "F6" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f6 = PhysicalKeyboardKey(0x0007003f, debugName: kReleaseMode ? null : 'F6');

  /// Represents the location of the "F7" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f7 = PhysicalKeyboardKey(0x00070040, debugName: kReleaseMode ? null : 'F7');

  /// Represents the location of the "F8" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f8 = PhysicalKeyboardKey(0x00070041, debugName: kReleaseMode ? null : 'F8');

  /// Represents the location of the "F9" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f9 = PhysicalKeyboardKey(0x00070042, debugName: kReleaseMode ? null : 'F9');

  /// Represents the location of the "F10" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f10 = PhysicalKeyboardKey(0x00070043, debugName: kReleaseMode ? null : 'F10');

  /// Represents the location of the "F11" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f11 = PhysicalKeyboardKey(0x00070044, debugName: kReleaseMode ? null : 'F11');

  /// Represents the location of the "F12" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f12 = PhysicalKeyboardKey(0x00070045, debugName: kReleaseMode ? null : 'F12');

  /// Represents the location of the "Print Screen" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey printScreen = PhysicalKeyboardKey(0x00070046, debugName: kReleaseMode ? null : 'Print Screen');

  /// Represents the location of the "Scroll Lock" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey scrollLock = PhysicalKeyboardKey(0x00070047, debugName: kReleaseMode ? null : 'Scroll Lock');

  /// Represents the location of the "Pause" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey pause = PhysicalKeyboardKey(0x00070048, debugName: kReleaseMode ? null : 'Pause');

  /// Represents the location of the "Insert" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey insert = PhysicalKeyboardKey(0x00070049, debugName: kReleaseMode ? null : 'Insert');

  /// Represents the location of the "Home" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey home = PhysicalKeyboardKey(0x0007004a, debugName: kReleaseMode ? null : 'Home');

  /// Represents the location of the "Page Up" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey pageUp = PhysicalKeyboardKey(0x0007004b, debugName: kReleaseMode ? null : 'Page Up');

  /// Represents the location of the "Delete" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey delete = PhysicalKeyboardKey(0x0007004c, debugName: kReleaseMode ? null : 'Delete');

  /// Represents the location of the "End" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey end = PhysicalKeyboardKey(0x0007004d, debugName: kReleaseMode ? null : 'End');

  /// Represents the location of the "Page Down" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey pageDown = PhysicalKeyboardKey(0x0007004e, debugName: kReleaseMode ? null : 'Page Down');

  /// Represents the location of the "Arrow Right" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey arrowRight = PhysicalKeyboardKey(0x0007004f, debugName: kReleaseMode ? null : 'Arrow Right');

  /// Represents the location of the "Arrow Left" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey arrowLeft = PhysicalKeyboardKey(0x00070050, debugName: kReleaseMode ? null : 'Arrow Left');

  /// Represents the location of the "Arrow Down" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey arrowDown = PhysicalKeyboardKey(0x00070051, debugName: kReleaseMode ? null : 'Arrow Down');

  /// Represents the location of the "Arrow Up" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey arrowUp = PhysicalKeyboardKey(0x00070052, debugName: kReleaseMode ? null : 'Arrow Up');

  /// Represents the location of the "Num Lock" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numLock = PhysicalKeyboardKey(0x00070053, debugName: kReleaseMode ? null : 'Num Lock');

  /// Represents the location of the "Numpad Divide" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadDivide = PhysicalKeyboardKey(0x00070054, debugName: kReleaseMode ? null : 'Numpad Divide');

  /// Represents the location of the "Numpad Multiply" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadMultiply = PhysicalKeyboardKey(0x00070055, debugName: kReleaseMode ? null : 'Numpad Multiply');

  /// Represents the location of the "Numpad Subtract" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadSubtract = PhysicalKeyboardKey(0x00070056, debugName: kReleaseMode ? null : 'Numpad Subtract');

  /// Represents the location of the "Numpad Add" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadAdd = PhysicalKeyboardKey(0x00070057, debugName: kReleaseMode ? null : 'Numpad Add');

  /// Represents the location of the "Numpad Enter" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadEnter = PhysicalKeyboardKey(0x00070058, debugName: kReleaseMode ? null : 'Numpad Enter');

  /// Represents the location of the "Numpad 1" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad1 = PhysicalKeyboardKey(0x00070059, debugName: kReleaseMode ? null : 'Numpad 1');

  /// Represents the location of the "Numpad 2" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad2 = PhysicalKeyboardKey(0x0007005a, debugName: kReleaseMode ? null : 'Numpad 2');

  /// Represents the location of the "Numpad 3" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad3 = PhysicalKeyboardKey(0x0007005b, debugName: kReleaseMode ? null : 'Numpad 3');

  /// Represents the location of the "Numpad 4" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad4 = PhysicalKeyboardKey(0x0007005c, debugName: kReleaseMode ? null : 'Numpad 4');

  /// Represents the location of the "Numpad 5" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad5 = PhysicalKeyboardKey(0x0007005d, debugName: kReleaseMode ? null : 'Numpad 5');

  /// Represents the location of the "Numpad 6" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad6 = PhysicalKeyboardKey(0x0007005e, debugName: kReleaseMode ? null : 'Numpad 6');

  /// Represents the location of the "Numpad 7" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad7 = PhysicalKeyboardKey(0x0007005f, debugName: kReleaseMode ? null : 'Numpad 7');

  /// Represents the location of the "Numpad 8" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad8 = PhysicalKeyboardKey(0x00070060, debugName: kReleaseMode ? null : 'Numpad 8');

  /// Represents the location of the "Numpad 9" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad9 = PhysicalKeyboardKey(0x00070061, debugName: kReleaseMode ? null : 'Numpad 9');

  /// Represents the location of the "Numpad 0" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpad0 = PhysicalKeyboardKey(0x00070062, debugName: kReleaseMode ? null : 'Numpad 0');

  /// Represents the location of the "Numpad Decimal" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadDecimal = PhysicalKeyboardKey(0x00070063, debugName: kReleaseMode ? null : 'Numpad Decimal');

  /// Represents the location of the "Intl Backslash" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey intlBackslash = PhysicalKeyboardKey(0x00070064, debugName: kReleaseMode ? null : 'Intl Backslash');

  /// Represents the location of the "Context Menu" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey contextMenu = PhysicalKeyboardKey(0x00070065, debugName: kReleaseMode ? null : 'Context Menu');

  /// Represents the location of the "Power" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey power = PhysicalKeyboardKey(0x00070066, debugName: kReleaseMode ? null : 'Power');

  /// Represents the location of the "Numpad Equal" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadEqual = PhysicalKeyboardKey(0x00070067, debugName: kReleaseMode ? null : 'Numpad Equal');

  /// Represents the location of the "F13" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f13 = PhysicalKeyboardKey(0x00070068, debugName: kReleaseMode ? null : 'F13');

  /// Represents the location of the "F14" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f14 = PhysicalKeyboardKey(0x00070069, debugName: kReleaseMode ? null : 'F14');

  /// Represents the location of the "F15" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f15 = PhysicalKeyboardKey(0x0007006a, debugName: kReleaseMode ? null : 'F15');

  /// Represents the location of the "F16" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f16 = PhysicalKeyboardKey(0x0007006b, debugName: kReleaseMode ? null : 'F16');

  /// Represents the location of the "F17" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f17 = PhysicalKeyboardKey(0x0007006c, debugName: kReleaseMode ? null : 'F17');

  /// Represents the location of the "F18" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f18 = PhysicalKeyboardKey(0x0007006d, debugName: kReleaseMode ? null : 'F18');

  /// Represents the location of the "F19" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f19 = PhysicalKeyboardKey(0x0007006e, debugName: kReleaseMode ? null : 'F19');

  /// Represents the location of the "F20" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f20 = PhysicalKeyboardKey(0x0007006f, debugName: kReleaseMode ? null : 'F20');

  /// Represents the location of the "F21" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f21 = PhysicalKeyboardKey(0x00070070, debugName: kReleaseMode ? null : 'F21');

  /// Represents the location of the "F22" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f22 = PhysicalKeyboardKey(0x00070071, debugName: kReleaseMode ? null : 'F22');

  /// Represents the location of the "F23" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f23 = PhysicalKeyboardKey(0x00070072, debugName: kReleaseMode ? null : 'F23');

  /// Represents the location of the "F24" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey f24 = PhysicalKeyboardKey(0x00070073, debugName: kReleaseMode ? null : 'F24');

  /// Represents the location of the "Open" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey open = PhysicalKeyboardKey(0x00070074, debugName: kReleaseMode ? null : 'Open');

  /// Represents the location of the "Help" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey help = PhysicalKeyboardKey(0x00070075, debugName: kReleaseMode ? null : 'Help');

  /// Represents the location of the "Select" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey select = PhysicalKeyboardKey(0x00070077, debugName: kReleaseMode ? null : 'Select');

  /// Represents the location of the "Again" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey again = PhysicalKeyboardKey(0x00070079, debugName: kReleaseMode ? null : 'Again');

  /// Represents the location of the "Undo" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey undo = PhysicalKeyboardKey(0x0007007a, debugName: kReleaseMode ? null : 'Undo');

  /// Represents the location of the "Cut" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey cut = PhysicalKeyboardKey(0x0007007b, debugName: kReleaseMode ? null : 'Cut');

  /// Represents the location of the "Copy" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey copy = PhysicalKeyboardKey(0x0007007c, debugName: kReleaseMode ? null : 'Copy');

  /// Represents the location of the "Paste" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey paste = PhysicalKeyboardKey(0x0007007d, debugName: kReleaseMode ? null : 'Paste');

  /// Represents the location of the "Find" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey find = PhysicalKeyboardKey(0x0007007e, debugName: kReleaseMode ? null : 'Find');

  /// Represents the location of the "Audio Volume Mute" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey audioVolumeMute = PhysicalKeyboardKey(0x0007007f, debugName: kReleaseMode ? null : 'Audio Volume Mute');

  /// Represents the location of the "Audio Volume Up" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey audioVolumeUp = PhysicalKeyboardKey(0x00070080, debugName: kReleaseMode ? null : 'Audio Volume Up');

  /// Represents the location of the "Audio Volume Down" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey audioVolumeDown = PhysicalKeyboardKey(0x00070081, debugName: kReleaseMode ? null : 'Audio Volume Down');

  /// Represents the location of the "Numpad Comma" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadComma = PhysicalKeyboardKey(0x00070085, debugName: kReleaseMode ? null : 'Numpad Comma');

  /// Represents the location of the "Intl Ro" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey intlRo = PhysicalKeyboardKey(0x00070087, debugName: kReleaseMode ? null : 'Intl Ro');

  /// Represents the location of the "Kana Mode" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey kanaMode = PhysicalKeyboardKey(0x00070088, debugName: kReleaseMode ? null : 'Kana Mode');

  /// Represents the location of the "Intl Yen" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey intlYen = PhysicalKeyboardKey(0x00070089, debugName: kReleaseMode ? null : 'Intl Yen');

  /// Represents the location of the "Convert" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey convert = PhysicalKeyboardKey(0x0007008a, debugName: kReleaseMode ? null : 'Convert');

  /// Represents the location of the "Non Convert" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey nonConvert = PhysicalKeyboardKey(0x0007008b, debugName: kReleaseMode ? null : 'Non Convert');

  /// Represents the location of the "Lang 1" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey lang1 = PhysicalKeyboardKey(0x00070090, debugName: kReleaseMode ? null : 'Lang 1');

  /// Represents the location of the "Lang 2" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey lang2 = PhysicalKeyboardKey(0x00070091, debugName: kReleaseMode ? null : 'Lang 2');

  /// Represents the location of the "Lang 3" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey lang3 = PhysicalKeyboardKey(0x00070092, debugName: kReleaseMode ? null : 'Lang 3');

  /// Represents the location of the "Lang 4" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey lang4 = PhysicalKeyboardKey(0x00070093, debugName: kReleaseMode ? null : 'Lang 4');

  /// Represents the location of the "Lang 5" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey lang5 = PhysicalKeyboardKey(0x00070094, debugName: kReleaseMode ? null : 'Lang 5');

  /// Represents the location of the "Abort" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey abort = PhysicalKeyboardKey(0x0007009b, debugName: kReleaseMode ? null : 'Abort');

  /// Represents the location of the "Props" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey props = PhysicalKeyboardKey(0x000700a3, debugName: kReleaseMode ? null : 'Props');

  /// Represents the location of the "Numpad Paren Left" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadParenLeft = PhysicalKeyboardKey(0x000700b6, debugName: kReleaseMode ? null : 'Numpad Paren Left');

  /// Represents the location of the "Numpad Paren Right" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadParenRight = PhysicalKeyboardKey(0x000700b7, debugName: kReleaseMode ? null : 'Numpad Paren Right');

  /// Represents the location of the "Numpad Backspace" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadBackspace = PhysicalKeyboardKey(0x000700bb, debugName: kReleaseMode ? null : 'Numpad Backspace');

  /// Represents the location of the "Numpad Memory Store" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadMemoryStore = PhysicalKeyboardKey(0x000700d0, debugName: kReleaseMode ? null : 'Numpad Memory Store');

  /// Represents the location of the "Numpad Memory Recall" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadMemoryRecall = PhysicalKeyboardKey(0x000700d1, debugName: kReleaseMode ? null : 'Numpad Memory Recall');

  /// Represents the location of the "Numpad Memory Clear" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadMemoryClear = PhysicalKeyboardKey(0x000700d2, debugName: kReleaseMode ? null : 'Numpad Memory Clear');

  /// Represents the location of the "Numpad Memory Add" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadMemoryAdd = PhysicalKeyboardKey(0x000700d3, debugName: kReleaseMode ? null : 'Numpad Memory Add');

  /// Represents the location of the "Numpad Memory Subtract" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadMemorySubtract = PhysicalKeyboardKey(0x000700d4, debugName: kReleaseMode ? null : 'Numpad Memory Subtract');

  /// Represents the location of the "Numpad Sign Change" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadSignChange = PhysicalKeyboardKey(0x000700d7, debugName: kReleaseMode ? null : 'Numpad Sign Change');

  /// Represents the location of the "Numpad Clear" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadClear = PhysicalKeyboardKey(0x000700d8, debugName: kReleaseMode ? null : 'Numpad Clear');

  /// Represents the location of the "Numpad Clear Entry" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey numpadClearEntry = PhysicalKeyboardKey(0x000700d9, debugName: kReleaseMode ? null : 'Numpad Clear Entry');

  /// Represents the location of the "Control Left" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey controlLeft = PhysicalKeyboardKey(0x000700e0, debugName: kReleaseMode ? null : 'Control Left');

  /// Represents the location of the "Shift Left" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey shiftLeft = PhysicalKeyboardKey(0x000700e1, debugName: kReleaseMode ? null : 'Shift Left');

  /// Represents the location of the "Alt Left" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey altLeft = PhysicalKeyboardKey(0x000700e2, debugName: kReleaseMode ? null : 'Alt Left');

  /// Represents the location of the "Meta Left" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey metaLeft = PhysicalKeyboardKey(0x000700e3, debugName: kReleaseMode ? null : 'Meta Left');

  /// Represents the location of the "Control Right" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey controlRight = PhysicalKeyboardKey(0x000700e4, debugName: kReleaseMode ? null : 'Control Right');

  /// Represents the location of the "Shift Right" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey shiftRight = PhysicalKeyboardKey(0x000700e5, debugName: kReleaseMode ? null : 'Shift Right');

  /// Represents the location of the "Alt Right" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey altRight = PhysicalKeyboardKey(0x000700e6, debugName: kReleaseMode ? null : 'Alt Right');

  /// Represents the location of the "Meta Right" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey metaRight = PhysicalKeyboardKey(0x000700e7, debugName: kReleaseMode ? null : 'Meta Right');

  /// Represents the location of the "Info" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey info = PhysicalKeyboardKey(0x000c0060, debugName: kReleaseMode ? null : 'Info');

  /// Represents the location of the "Closed Caption Toggle" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey closedCaptionToggle = PhysicalKeyboardKey(0x000c0061, debugName: kReleaseMode ? null : 'Closed Caption Toggle');

  /// Represents the location of the "Brightness Up" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey brightnessUp = PhysicalKeyboardKey(0x000c006f, debugName: kReleaseMode ? null : 'Brightness Up');

  /// Represents the location of the "Brightness Down" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey brightnessDown = PhysicalKeyboardKey(0x000c0070, debugName: kReleaseMode ? null : 'Brightness Down');

  /// Represents the location of the "Brightness Toggle" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey brightnessToggle = PhysicalKeyboardKey(0x000c0072, debugName: kReleaseMode ? null : 'Brightness Toggle');

  /// Represents the location of the "Brightness Minimum" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey brightnessMinimum = PhysicalKeyboardKey(0x000c0073, debugName: kReleaseMode ? null : 'Brightness Minimum');

  /// Represents the location of the "Brightness Maximum" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey brightnessMaximum = PhysicalKeyboardKey(0x000c0074, debugName: kReleaseMode ? null : 'Brightness Maximum');

  /// Represents the location of the "Brightness Auto" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey brightnessAuto = PhysicalKeyboardKey(0x000c0075, debugName: kReleaseMode ? null : 'Brightness Auto');

  /// Represents the location of the "Media Last" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaLast = PhysicalKeyboardKey(0x000c0083, debugName: kReleaseMode ? null : 'Media Last');

  /// Represents the location of the "Launch Phone" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchPhone = PhysicalKeyboardKey(0x000c008c, debugName: kReleaseMode ? null : 'Launch Phone');

  /// Represents the location of the "Program Guide" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey programGuide = PhysicalKeyboardKey(0x000c008d, debugName: kReleaseMode ? null : 'Program Guide');

  /// Represents the location of the "Exit" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey exit = PhysicalKeyboardKey(0x000c0094, debugName: kReleaseMode ? null : 'Exit');

  /// Represents the location of the "Channel Up" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey channelUp = PhysicalKeyboardKey(0x000c009c, debugName: kReleaseMode ? null : 'Channel Up');

  /// Represents the location of the "Channel Down" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey channelDown = PhysicalKeyboardKey(0x000c009d, debugName: kReleaseMode ? null : 'Channel Down');

  /// Represents the location of the "Media Play" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaPlay = PhysicalKeyboardKey(0x000c00b0, debugName: kReleaseMode ? null : 'Media Play');

  /// Represents the location of the "Media Record" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaRecord = PhysicalKeyboardKey(0x000c00b2, debugName: kReleaseMode ? null : 'Media Record');

  /// Represents the location of the "Media Fast Forward" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaFastForward = PhysicalKeyboardKey(0x000c00b3, debugName: kReleaseMode ? null : 'Media Fast Forward');

  /// Represents the location of the "Media Rewind" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaRewind = PhysicalKeyboardKey(0x000c00b4, debugName: kReleaseMode ? null : 'Media Rewind');

  /// Represents the location of the "Media Track Next" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaTrackNext = PhysicalKeyboardKey(0x000c00b5, debugName: kReleaseMode ? null : 'Media Track Next');

  /// Represents the location of the "Media Track Previous" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaTrackPrevious = PhysicalKeyboardKey(0x000c00b6, debugName: kReleaseMode ? null : 'Media Track Previous');

  /// Represents the location of the "Media Stop" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaStop = PhysicalKeyboardKey(0x000c00b7, debugName: kReleaseMode ? null : 'Media Stop');

  /// Represents the location of the "Eject" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey eject = PhysicalKeyboardKey(0x000c00b8, debugName: kReleaseMode ? null : 'Eject');

  /// Represents the location of the "Media Play Pause" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaPlayPause = PhysicalKeyboardKey(0x000c00cd, debugName: kReleaseMode ? null : 'Media Play Pause');

  /// Represents the location of the "Speech Input Toggle" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey speechInputToggle = PhysicalKeyboardKey(0x000c00cf, debugName: kReleaseMode ? null : 'Speech Input Toggle');

  /// Represents the location of the "Bass Boost" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey bassBoost = PhysicalKeyboardKey(0x000c00e5, debugName: kReleaseMode ? null : 'Bass Boost');

  /// Represents the location of the "Media Select" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mediaSelect = PhysicalKeyboardKey(0x000c0183, debugName: kReleaseMode ? null : 'Media Select');

  /// Represents the location of the "Launch Word Processor" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchWordProcessor = PhysicalKeyboardKey(0x000c0184, debugName: kReleaseMode ? null : 'Launch Word Processor');

  /// Represents the location of the "Launch Spreadsheet" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchSpreadsheet = PhysicalKeyboardKey(0x000c0186, debugName: kReleaseMode ? null : 'Launch Spreadsheet');

  /// Represents the location of the "Launch Mail" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchMail = PhysicalKeyboardKey(0x000c018a, debugName: kReleaseMode ? null : 'Launch Mail');

  /// Represents the location of the "Launch Contacts" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchContacts = PhysicalKeyboardKey(0x000c018d, debugName: kReleaseMode ? null : 'Launch Contacts');

  /// Represents the location of the "Launch Calendar" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchCalendar = PhysicalKeyboardKey(0x000c018e, debugName: kReleaseMode ? null : 'Launch Calendar');

  /// Represents the location of the "Launch App2" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchApp2 = PhysicalKeyboardKey(0x000c0192, debugName: kReleaseMode ? null : 'Launch App2');

  /// Represents the location of the "Launch App1" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchApp1 = PhysicalKeyboardKey(0x000c0194, debugName: kReleaseMode ? null : 'Launch App1');

  /// Represents the location of the "Launch Internet Browser" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchInternetBrowser = PhysicalKeyboardKey(0x000c0196, debugName: kReleaseMode ? null : 'Launch Internet Browser');

  /// Represents the location of the "Log Off" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey logOff = PhysicalKeyboardKey(0x000c019c, debugName: kReleaseMode ? null : 'Log Off');

  /// Represents the location of the "Lock Screen" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey lockScreen = PhysicalKeyboardKey(0x000c019e, debugName: kReleaseMode ? null : 'Lock Screen');

  /// Represents the location of the "Launch Control Panel" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchControlPanel = PhysicalKeyboardKey(0x000c019f, debugName: kReleaseMode ? null : 'Launch Control Panel');

  /// Represents the location of the "Select Task" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey selectTask = PhysicalKeyboardKey(0x000c01a2, debugName: kReleaseMode ? null : 'Select Task');

  /// Represents the location of the "Launch Documents" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchDocuments = PhysicalKeyboardKey(0x000c01a7, debugName: kReleaseMode ? null : 'Launch Documents');

  /// Represents the location of the "Spell Check" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey spellCheck = PhysicalKeyboardKey(0x000c01ab, debugName: kReleaseMode ? null : 'Spell Check');

  /// Represents the location of the "Launch Keyboard Layout" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchKeyboardLayout = PhysicalKeyboardKey(0x000c01ae, debugName: kReleaseMode ? null : 'Launch Keyboard Layout');

  /// Represents the location of the "Launch Screen Saver" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchScreenSaver = PhysicalKeyboardKey(0x000c01b1, debugName: kReleaseMode ? null : 'Launch Screen Saver');

  /// Represents the location of the "Launch Assistant" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchAssistant = PhysicalKeyboardKey(0x000c01cb, debugName: kReleaseMode ? null : 'Launch Assistant');

  /// Represents the location of the "Launch Audio Browser" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey launchAudioBrowser = PhysicalKeyboardKey(0x000c01b7, debugName: kReleaseMode ? null : 'Launch Audio Browser');

  /// Represents the location of the "New Key" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey newKey = PhysicalKeyboardKey(0x000c0201, debugName: kReleaseMode ? null : 'New Key');

  /// Represents the location of the "Close" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey close = PhysicalKeyboardKey(0x000c0203, debugName: kReleaseMode ? null : 'Close');

  /// Represents the location of the "Save" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey save = PhysicalKeyboardKey(0x000c0207, debugName: kReleaseMode ? null : 'Save');

  /// Represents the location of the "Print" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey print = PhysicalKeyboardKey(0x000c0208, debugName: kReleaseMode ? null : 'Print');

  /// Represents the location of the "Browser Search" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey browserSearch = PhysicalKeyboardKey(0x000c0221, debugName: kReleaseMode ? null : 'Browser Search');

  /// Represents the location of the "Browser Home" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey browserHome = PhysicalKeyboardKey(0x000c0223, debugName: kReleaseMode ? null : 'Browser Home');

  /// Represents the location of the "Browser Back" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey browserBack = PhysicalKeyboardKey(0x000c0224, debugName: kReleaseMode ? null : 'Browser Back');

  /// Represents the location of the "Browser Forward" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey browserForward = PhysicalKeyboardKey(0x000c0225, debugName: kReleaseMode ? null : 'Browser Forward');

  /// Represents the location of the "Browser Stop" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey browserStop = PhysicalKeyboardKey(0x000c0226, debugName: kReleaseMode ? null : 'Browser Stop');

  /// Represents the location of the "Browser Refresh" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey browserRefresh = PhysicalKeyboardKey(0x000c0227, debugName: kReleaseMode ? null : 'Browser Refresh');

  /// Represents the location of the "Browser Favorites" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey browserFavorites = PhysicalKeyboardKey(0x000c022a, debugName: kReleaseMode ? null : 'Browser Favorites');

  /// Represents the location of the "Zoom In" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey zoomIn = PhysicalKeyboardKey(0x000c022d, debugName: kReleaseMode ? null : 'Zoom In');

  /// Represents the location of the "Zoom Out" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey zoomOut = PhysicalKeyboardKey(0x000c022e, debugName: kReleaseMode ? null : 'Zoom Out');

  /// Represents the location of the "Zoom Toggle" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey zoomToggle = PhysicalKeyboardKey(0x000c0232, debugName: kReleaseMode ? null : 'Zoom Toggle');

  /// Represents the location of the "Redo" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey redo = PhysicalKeyboardKey(0x000c0279, debugName: kReleaseMode ? null : 'Redo');

  /// Represents the location of the "Mail Reply" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mailReply = PhysicalKeyboardKey(0x000c0289, debugName: kReleaseMode ? null : 'Mail Reply');

  /// Represents the location of the "Mail Forward" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mailForward = PhysicalKeyboardKey(0x000c028b, debugName: kReleaseMode ? null : 'Mail Forward');

  /// Represents the location of the "Mail Send" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey mailSend = PhysicalKeyboardKey(0x000c028c, debugName: kReleaseMode ? null : 'Mail Send');

  /// Represents the location of the "Keyboard Layout Select" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey keyboardLayoutSelect = PhysicalKeyboardKey(0x000c029d, debugName: kReleaseMode ? null : 'Keyboard Layout Select');

  /// Represents the location of the "Show All Windows" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey showAllWindows = PhysicalKeyboardKey(0x000c029f, debugName: kReleaseMode ? null : 'Show All Windows');

  /// Represents the location of the "Game Button 1" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton1 = PhysicalKeyboardKey(0x0005ff01, debugName: kReleaseMode ? null : 'Game Button 1');

  /// Represents the location of the "Game Button 2" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton2 = PhysicalKeyboardKey(0x0005ff02, debugName: kReleaseMode ? null : 'Game Button 2');

  /// Represents the location of the "Game Button 3" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton3 = PhysicalKeyboardKey(0x0005ff03, debugName: kReleaseMode ? null : 'Game Button 3');

  /// Represents the location of the "Game Button 4" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton4 = PhysicalKeyboardKey(0x0005ff04, debugName: kReleaseMode ? null : 'Game Button 4');

  /// Represents the location of the "Game Button 5" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton5 = PhysicalKeyboardKey(0x0005ff05, debugName: kReleaseMode ? null : 'Game Button 5');

  /// Represents the location of the "Game Button 6" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton6 = PhysicalKeyboardKey(0x0005ff06, debugName: kReleaseMode ? null : 'Game Button 6');

  /// Represents the location of the "Game Button 7" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton7 = PhysicalKeyboardKey(0x0005ff07, debugName: kReleaseMode ? null : 'Game Button 7');

  /// Represents the location of the "Game Button 8" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton8 = PhysicalKeyboardKey(0x0005ff08, debugName: kReleaseMode ? null : 'Game Button 8');

  /// Represents the location of the "Game Button 9" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton9 = PhysicalKeyboardKey(0x0005ff09, debugName: kReleaseMode ? null : 'Game Button 9');

  /// Represents the location of the "Game Button 10" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton10 = PhysicalKeyboardKey(0x0005ff0a, debugName: kReleaseMode ? null : 'Game Button 10');

  /// Represents the location of the "Game Button 11" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton11 = PhysicalKeyboardKey(0x0005ff0b, debugName: kReleaseMode ? null : 'Game Button 11');

  /// Represents the location of the "Game Button 12" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton12 = PhysicalKeyboardKey(0x0005ff0c, debugName: kReleaseMode ? null : 'Game Button 12');

  /// Represents the location of the "Game Button 13" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton13 = PhysicalKeyboardKey(0x0005ff0d, debugName: kReleaseMode ? null : 'Game Button 13');

  /// Represents the location of the "Game Button 14" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton14 = PhysicalKeyboardKey(0x0005ff0e, debugName: kReleaseMode ? null : 'Game Button 14');

  /// Represents the location of the "Game Button 15" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton15 = PhysicalKeyboardKey(0x0005ff0f, debugName: kReleaseMode ? null : 'Game Button 15');

  /// Represents the location of the "Game Button 16" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButton16 = PhysicalKeyboardKey(0x0005ff10, debugName: kReleaseMode ? null : 'Game Button 16');

  /// Represents the location of the "Game Button A" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonA = PhysicalKeyboardKey(0x0005ff11, debugName: kReleaseMode ? null : 'Game Button A');

  /// Represents the location of the "Game Button B" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonB = PhysicalKeyboardKey(0x0005ff12, debugName: kReleaseMode ? null : 'Game Button B');

  /// Represents the location of the "Game Button C" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonC = PhysicalKeyboardKey(0x0005ff13, debugName: kReleaseMode ? null : 'Game Button C');

  /// Represents the location of the "Game Button Left 1" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonLeft1 = PhysicalKeyboardKey(0x0005ff14, debugName: kReleaseMode ? null : 'Game Button Left 1');

  /// Represents the location of the "Game Button Left 2" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonLeft2 = PhysicalKeyboardKey(0x0005ff15, debugName: kReleaseMode ? null : 'Game Button Left 2');

  /// Represents the location of the "Game Button Mode" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonMode = PhysicalKeyboardKey(0x0005ff16, debugName: kReleaseMode ? null : 'Game Button Mode');

  /// Represents the location of the "Game Button Right 1" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonRight1 = PhysicalKeyboardKey(0x0005ff17, debugName: kReleaseMode ? null : 'Game Button Right 1');

  /// Represents the location of the "Game Button Right 2" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonRight2 = PhysicalKeyboardKey(0x0005ff18, debugName: kReleaseMode ? null : 'Game Button Right 2');

  /// Represents the location of the "Game Button Select" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonSelect = PhysicalKeyboardKey(0x0005ff19, debugName: kReleaseMode ? null : 'Game Button Select');

  /// Represents the location of the "Game Button Start" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonStart = PhysicalKeyboardKey(0x0005ff1a, debugName: kReleaseMode ? null : 'Game Button Start');

  /// Represents the location of the "Game Button Thumb Left" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonThumbLeft = PhysicalKeyboardKey(0x0005ff1b, debugName: kReleaseMode ? null : 'Game Button Thumb Left');

  /// Represents the location of the "Game Button Thumb Right" key on a
  /// generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonThumbRight = PhysicalKeyboardKey(0x0005ff1c, debugName: kReleaseMode ? null : 'Game Button Thumb Right');

  /// Represents the location of the "Game Button X" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonX = PhysicalKeyboardKey(0x0005ff1d, debugName: kReleaseMode ? null : 'Game Button X');

  /// Represents the location of the "Game Button Y" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonY = PhysicalKeyboardKey(0x0005ff1e, debugName: kReleaseMode ? null : 'Game Button Y');

  /// Represents the location of the "Game Button Z" key on a generalized
  /// keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey gameButtonZ = PhysicalKeyboardKey(0x0005ff1f, debugName: kReleaseMode ? null : 'Game Button Z');

  /// Represents the location of the "Fn" key on a generalized keyboard.
  ///
  /// See the function [RawKeyEvent.physicalKey] for more information.
  static const PhysicalKeyboardKey fn = PhysicalKeyboardKey(0x00000012, debugName: kReleaseMode ? null : 'Fn');

  // A list of all the predefined constant PhysicalKeyboardKeys so that they
  // can be searched.
  static const Map<int, PhysicalKeyboardKey> _knownPhysicalKeys = <int, PhysicalKeyboardKey>{
    0x00000000: none,
    0x00000010: hyper,
    0x00000011: superKey,
    0x00000013: fnLock,
    0x00000014: suspend,
    0x00000015: resume,
    0x00000016: turbo,
    0x00010082: sleep,
    0x00010083: wakeUp,
    0x000100b5: displayToggleIntExt,
    0x00070000: usbReserved,
    0x00070001: usbErrorRollOver,
    0x00070002: usbPostFail,
    0x00070003: usbErrorUndefined,
    0x00070004: keyA,
    0x00070005: keyB,
    0x00070006: keyC,
    0x00070007: keyD,
    0x00070008: keyE,
    0x00070009: keyF,
    0x0007000a: keyG,
    0x0007000b: keyH,
    0x0007000c: keyI,
    0x0007000d: keyJ,
    0x0007000e: keyK,
    0x0007000f: keyL,
    0x00070010: keyM,
    0x00070011: keyN,
    0x00070012: keyO,
    0x00070013: keyP,
    0x00070014: keyQ,
    0x00070015: keyR,
    0x00070016: keyS,
    0x00070017: keyT,
    0x00070018: keyU,
    0x00070019: keyV,
    0x0007001a: keyW,
    0x0007001b: keyX,
    0x0007001c: keyY,
    0x0007001d: keyZ,
    0x0007001e: digit1,
    0x0007001f: digit2,
    0x00070020: digit3,
    0x00070021: digit4,
    0x00070022: digit5,
    0x00070023: digit6,
    0x00070024: digit7,
    0x00070025: digit8,
    0x00070026: digit9,
    0x00070027: digit0,
    0x00070028: enter,
    0x00070029: escape,
    0x0007002a: backspace,
    0x0007002b: tab,
    0x0007002c: space,
    0x0007002d: minus,
    0x0007002e: equal,
    0x0007002f: bracketLeft,
    0x00070030: bracketRight,
    0x00070031: backslash,
    0x00070033: semicolon,
    0x00070034: quote,
    0x00070035: backquote,
    0x00070036: comma,
    0x00070037: period,
    0x00070038: slash,
    0x00070039: capsLock,
    0x0007003a: f1,
    0x0007003b: f2,
    0x0007003c: f3,
    0x0007003d: f4,
    0x0007003e: f5,
    0x0007003f: f6,
    0x00070040: f7,
    0x00070041: f8,
    0x00070042: f9,
    0x00070043: f10,
    0x00070044: f11,
    0x00070045: f12,
    0x00070046: printScreen,
    0x00070047: scrollLock,
    0x00070048: pause,
    0x00070049: insert,
    0x0007004a: home,
    0x0007004b: pageUp,
    0x0007004c: delete,
    0x0007004d: end,
    0x0007004e: pageDown,
    0x0007004f: arrowRight,
    0x00070050: arrowLeft,
    0x00070051: arrowDown,
    0x00070052: arrowUp,
    0x00070053: numLock,
    0x00070054: numpadDivide,
    0x00070055: numpadMultiply,
    0x00070056: numpadSubtract,
    0x00070057: numpadAdd,
    0x00070058: numpadEnter,
    0x00070059: numpad1,
    0x0007005a: numpad2,
    0x0007005b: numpad3,
    0x0007005c: numpad4,
    0x0007005d: numpad5,
    0x0007005e: numpad6,
    0x0007005f: numpad7,
    0x00070060: numpad8,
    0x00070061: numpad9,
    0x00070062: numpad0,
    0x00070063: numpadDecimal,
    0x00070064: intlBackslash,
    0x00070065: contextMenu,
    0x00070066: power,
    0x00070067: numpadEqual,
    0x00070068: f13,
    0x00070069: f14,
    0x0007006a: f15,
    0x0007006b: f16,
    0x0007006c: f17,
    0x0007006d: f18,
    0x0007006e: f19,
    0x0007006f: f20,
    0x00070070: f21,
    0x00070071: f22,
    0x00070072: f23,
    0x00070073: f24,
    0x00070074: open,
    0x00070075: help,
    0x00070077: select,
    0x00070079: again,
    0x0007007a: undo,
    0x0007007b: cut,
    0x0007007c: copy,
    0x0007007d: paste,
    0x0007007e: find,
    0x0007007f: audioVolumeMute,
    0x00070080: audioVolumeUp,
    0x00070081: audioVolumeDown,
    0x00070085: numpadComma,
    0x00070087: intlRo,
    0x00070088: kanaMode,
    0x00070089: intlYen,
    0x0007008a: convert,
    0x0007008b: nonConvert,
    0x00070090: lang1,
    0x00070091: lang2,
    0x00070092: lang3,
    0x00070093: lang4,
    0x00070094: lang5,
    0x0007009b: abort,
    0x000700a3: props,
    0x000700b6: numpadParenLeft,
    0x000700b7: numpadParenRight,
    0x000700bb: numpadBackspace,
    0x000700d0: numpadMemoryStore,
    0x000700d1: numpadMemoryRecall,
    0x000700d2: numpadMemoryClear,
    0x000700d3: numpadMemoryAdd,
    0x000700d4: numpadMemorySubtract,
    0x000700d7: numpadSignChange,
    0x000700d8: numpadClear,
    0x000700d9: numpadClearEntry,
    0x000700e0: controlLeft,
    0x000700e1: shiftLeft,
    0x000700e2: altLeft,
    0x000700e3: metaLeft,
    0x000700e4: controlRight,
    0x000700e5: shiftRight,
    0x000700e6: altRight,
    0x000700e7: metaRight,
    0x000c0060: info,
    0x000c0061: closedCaptionToggle,
    0x000c006f: brightnessUp,
    0x000c0070: brightnessDown,
    0x000c0072: brightnessToggle,
    0x000c0073: brightnessMinimum,
    0x000c0074: brightnessMaximum,
    0x000c0075: brightnessAuto,
    0x000c0083: mediaLast,
    0x000c008c: launchPhone,
    0x000c008d: programGuide,
    0x000c0094: exit,
    0x000c009c: channelUp,
    0x000c009d: channelDown,
    0x000c00b0: mediaPlay,
    0x000c00b2: mediaRecord,
    0x000c00b3: mediaFastForward,
    0x000c00b4: mediaRewind,
    0x000c00b5: mediaTrackNext,
    0x000c00b6: mediaTrackPrevious,
    0x000c00b7: mediaStop,
    0x000c00b8: eject,
    0x000c00cd: mediaPlayPause,
    0x000c00cf: speechInputToggle,
    0x000c00e5: bassBoost,
    0x000c0183: mediaSelect,
    0x000c0184: launchWordProcessor,
    0x000c0186: launchSpreadsheet,
    0x000c018a: launchMail,
    0x000c018d: launchContacts,
    0x000c018e: launchCalendar,
    0x000c0192: launchApp2,
    0x000c0194: launchApp1,
    0x000c0196: launchInternetBrowser,
    0x000c019c: logOff,
    0x000c019e: lockScreen,
    0x000c019f: launchControlPanel,
    0x000c01a2: selectTask,
    0x000c01a7: launchDocuments,
    0x000c01ab: spellCheck,
    0x000c01ae: launchKeyboardLayout,
    0x000c01b1: launchScreenSaver,
    0x000c01cb: launchAssistant,
    0x000c01b7: launchAudioBrowser,
    0x000c0201: newKey,
    0x000c0203: close,
    0x000c0207: save,
    0x000c0208: print,
    0x000c0221: browserSearch,
    0x000c0223: browserHome,
    0x000c0224: browserBack,
    0x000c0225: browserForward,
    0x000c0226: browserStop,
    0x000c0227: browserRefresh,
    0x000c022a: browserFavorites,
    0x000c022d: zoomIn,
    0x000c022e: zoomOut,
    0x000c0232: zoomToggle,
    0x000c0279: redo,
    0x000c0289: mailReply,
    0x000c028b: mailForward,
    0x000c028c: mailSend,
    0x000c029d: keyboardLayoutSelect,
    0x000c029f: showAllWindows,
    0x0005ff01: gameButton1,
    0x0005ff02: gameButton2,
    0x0005ff03: gameButton3,
    0x0005ff04: gameButton4,
    0x0005ff05: gameButton5,
    0x0005ff06: gameButton6,
    0x0005ff07: gameButton7,
    0x0005ff08: gameButton8,
    0x0005ff09: gameButton9,
    0x0005ff0a: gameButton10,
    0x0005ff0b: gameButton11,
    0x0005ff0c: gameButton12,
    0x0005ff0d: gameButton13,
    0x0005ff0e: gameButton14,
    0x0005ff0f: gameButton15,
    0x0005ff10: gameButton16,
    0x0005ff11: gameButtonA,
    0x0005ff12: gameButtonB,
    0x0005ff13: gameButtonC,
    0x0005ff14: gameButtonLeft1,
    0x0005ff15: gameButtonLeft2,
    0x0005ff16: gameButtonMode,
    0x0005ff17: gameButtonRight1,
    0x0005ff18: gameButtonRight2,
    0x0005ff19: gameButtonSelect,
    0x0005ff1a: gameButtonStart,
    0x0005ff1b: gameButtonThumbLeft,
    0x0005ff1c: gameButtonThumbRight,
    0x0005ff1d: gameButtonX,
    0x0005ff1e: gameButtonY,
    0x0005ff1f: gameButtonZ,
    0x00000012: fn,
  };
}
