// Copyright 2014 The Flutter Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'dart:async';
import 'dart:math' as math;
import 'dart:typed_data';
import 'dart:ui' as ui;

import 'package:flutter/widgets.dart';
import 'package:vector_math/vector_math_64.dart';

import 'ink_well.dart';
import 'material.dart';

/// Begin a Material 3 ink sparkle ripple, centered at the tap or click position
/// relative to the [referenceBox].
///
/// This effect relies on a shader, and therefore hardware acceleration.
/// Currently, this is only supported by certain C++ engine platforms. The
/// platforms that are currently supported are Android, iOS, MacOS, Windows,
/// and Linux. Support for CanvasKit web can be tracked here:
///  - https://github.com/flutter/flutter/issues/85238
///
/// To use this effect, pass an instance of [splashFactory] to the
/// `splashFactory` parameter of either the Material [ThemeData] or any
/// component that has a `splashFactory` parameter, such as buttons:
///  - [ElevatedButton]
///  - [TextButton]
///  - [OutlinedButton]
///
/// The [controller] argument is typically obtained via
/// `Material.of(context)`.
///
/// If [containedInkWell] is true, then the effect will be sized to fit
/// the well rectangle, and clipped to it when drawn. The well
/// rectangle is the box returned by [rectCallback], if provided, or
/// otherwise is the bounds of the [referenceBox].
///
/// If [containedInkWell] is false, then [rectCallback] should be null.
/// The ink ripple is clipped only to the edges of the [Material].
/// This is the default.
///
/// When the ripple is removed, [onRemoved] will be called.
///
/// {@tool snippet}
///
/// For typical use, pass the [InkSparkle.splashFactory] to the `splashFactory`
/// parameter of a button style or [ThemeData].
///
/// ```dart
/// ElevatedButton(
///   style: ElevatedButton.styleFrom(splashFactory: InkSparkle.splashFactory),
///   child: const Text('Sparkle!'),
///   onPressed: () { },
/// )
/// ```
/// {@end-tool}
class InkSparkle extends InteractiveInkFeature {
  /// Begin a sparkly ripple effect, centered at [position] relative to
  /// [referenceBox].
  ///
  /// The [color] defines the color of the splash itself. The sparkles are
  /// always white.
  ///
  /// The [controller] argument is typically obtained via
  /// `Material.of(context)`.
  ///
  /// [textDirection] is used by [customBorder] if it is non-null. This allows
  /// the [customBorder]'s path to be properly defined if it was the path was
  /// expressed in terms of "start" and "end" instead of
  /// "left" and "right".
  ///
  /// If [containedInkWell] is true, then the ripple will be sized to fit
  /// the well rectangle, then clipped to it when drawn. The well
  /// rectangle is the box returned by [rectCallback], if provided, or
  /// otherwise is the bounds of the [referenceBox].
  ///
  /// If [containedInkWell] is false, then [rectCallback] should be null.
  /// The ink ripple is clipped only to the edges of the [Material].
  /// This is the default.
  ///
  /// Clipping can happen in 3 different ways:
  ///  1. If [customBorder] is provided, it is used to determine the path for
  ///     clipping.
  ///  2. If [customBorder] is null, and [borderRadius] is provided, then the
  ///     canvas is clipped by an [RRect] created from [borderRadius].
  ///  3. If [borderRadius] is the default [BorderRadius.zero], then the canvas
  ///     is clipped with [rectCallback].
  /// When the ripple is removed, [onRemoved] will be called.
  ///
  /// [turbulenceSeed] can be passed if a non random seed shold be used for
  /// the turbulence and sparkles. By default, the seed is a random number
  /// between 0.0 and 1000.0.
  ///
  /// Turbulence is an input to the shader and helps to provides a more natural,
  ///  non-circular, "splash" effect.
  ///
  /// Sparkle randomization is also driven by the [turbulenceSeed]. Sparkles are
  /// identified in the shader as "noise", and the sparkles are derived from
  /// pseudorandom triangular noise.
  InkSparkle({
    required MaterialInkController controller,
    required RenderBox referenceBox,
    required Color color,
    required Offset position,
    required TextDirection textDirection,
    bool containedInkWell = true,
    RectCallback? rectCallback,
    BorderRadius? borderRadius,
    ShapeBorder? customBorder,
    double? radius,
    VoidCallback? onRemoved,
    double? turbulenceSeed,
  }) : assert(containedInkWell || rectCallback == null),
       _color = color,
       _position = position,
       _borderRadius = borderRadius ?? BorderRadius.zero,
       _customBorder = customBorder,
       _textDirection = textDirection,
       _targetRadius = (radius ?? _getTargetRadius(referenceBox, containedInkWell, rectCallback, position)) * _targetRadiusMultiplier,
       _clipCallback = _getClipCallback(referenceBox, containedInkWell, rectCallback),
       super(controller: controller, referenceBox: referenceBox, color: color, onRemoved: onRemoved) {
    // InkSparkle will not be painted until the async compilation completes.
    _InkSparkleFactory.compileShaderIfNeccessary();
    controller.addInkFeature(this);

    // All animation values are derived from Android 12 source code. See:
    // - https://cs.android.com/android/platform/superproject/+/master:frameworks/base/graphics/java/android/graphics/drawable/RippleShader.java
    // - https://cs.android.com/android/platform/superproject/+/master:frameworks/base/graphics/java/android/graphics/drawable/RippleDrawable.java
    // - https://cs.android.com/android/platform/superproject/+/master:frameworks/base/graphics/java/android/graphics/drawable/RippleAnimationSession.java

    // Immediately begin animating the ink.
    _animationController = AnimationController(
      duration: _animationDuration,
      vsync: controller.vsync,
    )..addListener(controller.markNeedsPaint)..forward();

    _radiusScale = TweenSequence<double>(
      <TweenSequenceItem<double>>[
        TweenSequenceItem<double>(
          tween: CurveTween(curve: Curves.fastOutSlowIn),
          weight: 75,
        ),
        TweenSequenceItem<double>(
          tween: ConstantTween<double>(1.0),
          weight: 25,
        ),
      ],
    ).animate(_animationController);

    // Functionally equivalent to Android 12's SkSL:
    //`return mix(u_touch, u_resolution, saturate(in_radius_scale * 2.0))`
    final Tween<Vector2> centerTween = Tween<Vector2>(
      begin: Vector2.array(<double>[_position.dx, _position.dy]),
      end: Vector2.array(<double>[referenceBox.size.width / 2, referenceBox.size.height / 2]),
    );
    final Animation<double> centerProgress = TweenSequence<double>(
      <TweenSequenceItem<double>>[
        TweenSequenceItem<double>(
          tween: Tween<double>(begin: 0.0, end: 1.0),
          weight: 50,
        ),
        TweenSequenceItem<double>(
          tween: ConstantTween<double>(1.0),
          weight: 50,
        ),
      ],
    ).animate(_radiusScale);
    _center = centerTween.animate(centerProgress);

    _alpha = TweenSequence<double>(
      <TweenSequenceItem<double>>[
        TweenSequenceItem<double>(
          tween: Tween<double>(begin: 0.0, end: 1.0),
          weight: 13,
        ),
        TweenSequenceItem<double>(
          tween: ConstantTween<double>(1.0),
          weight: 27,
        ),
        TweenSequenceItem<double>(
          tween: Tween<double>(begin: 1.0, end: 0.0),
          weight: 60,
        ),
      ],
    ).animate(_animationController);

    _sparkleAlpha = TweenSequence<double>(
      <TweenSequenceItem<double>>[
        TweenSequenceItem<double>(
          tween: Tween<double>(begin: 0.0, end: 1.0),
          weight: 13,
        ),
        TweenSequenceItem<double>(
          tween: ConstantTween<double>(1.0),
          weight: 27,
        ),
        TweenSequenceItem<double>(
          tween: Tween<double>(begin: 1.0, end: 0.0),
          weight: 50,
        ),
      ],
    ).animate(_animationController);

    // Creates an element of randomness so that ink eminating from the same
    // pixel have slightly different rings and sparkles.
    _turbulenceSeed = turbulenceSeed ?? math.Random().nextDouble() * 1000.0;
  }

  static const Duration _animationDuration = Duration(milliseconds: 617);
  static const double _targetRadiusMultiplier = 2.3;
  static const double _rotateRight = math.pi * 0.0078125;
  static const double _rotateLeft = -_rotateRight;
  static const double _noiseDensity = 2.1;

  late AnimationController _animationController;

  // The Android 12 version has these values calculated in the GLSL. They are
  // constant for every pixel in the animation, so the Flutter implementation
  // computes these animation values in software in order to simplify the shader
  // implementation and provide better performance on most devices.
  late Animation<Vector2> _center;
  late Animation<double> _radiusScale;
  late Animation<double> _alpha;
  late Animation<double> _sparkleAlpha;

  late double _turbulenceSeed;

  final Color _color;
  final Offset _position;
  final BorderRadius _borderRadius;
  final ShapeBorder? _customBorder;
  final double _targetRadius;
  final RectCallback? _clipCallback;
  final TextDirection _textDirection;

  /// Used to specify this type of ink splash for an [InkWell], [InkResponse],
  /// material [Theme], or [ButtonStyle].
  ///
  /// Since no [turbulenceSeed] is passed, the effect will be random for
  /// subsequent presses in the same position.
  static const InteractiveInkFeatureFactory splashFactory = _InkSparkleFactory();

  /// Used to specify this type of ink splash for an [InkWell], [InkResponse],
  /// material [Theme], or [ButtonStyle].
  ///
  /// Since a [turbulenceSeed] is passed, the effect will not be random for
  /// subsequent presses in the same position. This can be used for testing.
  static const InteractiveInkFeatureFactory constantTurbulenceSeedSplashFactory = _InkSparkleFactory.constantTurbulenceSeed();

  @override
  void dispose() {
    _animationController.stop();
    _animationController.dispose();
    super.dispose();
  }

  @override
  void paintFeature(Canvas canvas, Matrix4 transform) {
    // InkSparkle can only paint if its shader has been compiled.
    if (_InkSparkleFactory._shaderManager == null) {
      // Skipping paintFeature because the shader it relies on is not ready to
      // be used. InkSparkleFactory.compileShaderIfNeccessary must complete
      // before InkSparkle can paint.
      return;
    }

    canvas.save();
    _transformCanvas(canvas: canvas, transform: transform);
    if (_clipCallback != null) {
      _clipCanvas(
        canvas: canvas,
        clipCallback: _clipCallback!,
        textDirection: _textDirection,
        customBorder: _customBorder,
        borderRadius: _borderRadius,
      );
    }

    final Paint paint = Paint()..shader = _createRippleShader();
    if (_clipCallback != null) {
      canvas.drawRect(_clipCallback!(), paint);
    } else {
      canvas.drawPaint(paint);
    }
    canvas.restore();
  }

  double get _width => referenceBox.size.width;
  double get _height => referenceBox.size.height;

  /// All double values for uniforms come from the Android 12 ripple
  /// implentation from the following files:
  /// - https://cs.android.com/android/platform/superproject/+/master:frameworks/base/graphics/java/android/graphics/drawable/RippleShader.java
  /// - https://cs.android.com/android/platform/superproject/+/master:frameworks/base/graphics/java/android/graphics/drawable/RippleDrawable.java
  /// - https://cs.android.com/android/platform/superproject/+/master:frameworks/base/graphics/java/android/graphics/drawable/RippleAnimationSession.java
  Shader _createRippleShader() {
    const double turbulenceScale = 1.5;
    final double turbulencePhase = _turbulenceSeed + _radiusScale.value;
    final double noisePhase = turbulencePhase;
    final double rotation1 = turbulencePhase * _rotateRight + 1.7 * math.pi;
    final double rotation2 = turbulencePhase * _rotateLeft + 2.0 * math.pi;
    final double rotation3 = turbulencePhase * _rotateRight + 2.75 * math.pi;

    return _InkSparkleFactory._shaderManager!.shader(
      // The following uniforms are the same throughout the animation.
      uColor: _colorToVector4(_color),
      uSparkleColor: Vector4(1.0, 1.0, 1.0, 1.0),
      uBlur: 1.0,
      uMaxRadius: _targetRadius,
      uResolutionScale: Vector2(1.0 / _width, 1.0 / _height),
      uNoiseScale: Vector2(_noiseDensity / _width, _noiseDensity / _height),

      // The following uniforms update each frame of the animation.
      uCenter: _center.value,
      uRadiusScale: _radiusScale.value,
      uAlpha: _alpha.value,
      uSparkleAlpha: _sparkleAlpha.value,

      // The following uniforms are driven by the turbulence phase and change
      // each frame of the animation. Theese unifroms are uses to modify the
      // default (if these fields are unset or 0) circular outer ring to a
      // non-uniform shape that is more like an actual ink splash. In addition
      // to the positional based triangular noise created in the shader, these
      // uniforms also vary the appearence of the sparkles even when the same
      // location is tapped.
      uTurbulencePhase: turbulencePhase,
      uNoisePhase: noisePhase / 1000.0,
      uCircle1: Vector2(
        turbulenceScale * 0.5 + (turbulencePhase * 0.01 * math.cos(turbulenceScale * 0.55)),
        turbulenceScale * 0.5 + (turbulencePhase * 0.01 * math.sin(turbulenceScale * 0.55)),
      ),
      uCircle2: Vector2(
        turbulenceScale * 0.2 + (turbulencePhase * -0.0066 * math.cos(turbulenceScale * 0.45)),
        turbulenceScale * 0.2 + (turbulencePhase * -0.0066 * math.sin(turbulenceScale * 0.45)),
      ),
      uCircle3: Vector2(
        turbulenceScale + (turbulencePhase * -0.0066 * math.cos(turbulenceScale * 0.35)),
        turbulenceScale + (turbulencePhase * -0.0066 * math.sin(turbulenceScale * 0.35)),
      ),
      uRotation1: Vector2(math.cos(rotation1), math.sin(rotation1)),
      uRotation2: Vector2(math.cos(rotation2), math.sin(rotation2)),
      uRotation3: Vector2(math.cos(rotation3), math.sin(rotation3)),
    );
  }

  Vector4 _colorToVector4(Color color) {
    return Vector4(
        color.red / 255.0,
        color.blue / 255.0,
        color.green / 255.0,
        color.alpha / 255.0,
      );
  }

  /// Transforms the canvas for an ink feature to be painted on the [canvas].
  ///
  /// This should be called before painting ink features that do not use
  /// [paintInkCircle].
  ///
  /// The [transform] argument is the [Matrix4] transform that typically
  /// shifts the coordinate space of the canvas to the space in which
  /// the ink feature is to be painted.
  ///
  /// For examples on how the function is used, see [InkSparkle] and [paintInkCircle].
  void _transformCanvas({
    required Canvas canvas,
    required Matrix4 transform,
  }) {
    final Offset? originOffset = MatrixUtils.getAsTranslation(transform);
    if (originOffset == null) {
      canvas.transform(transform.storage);
    } else {
      canvas.translate(originOffset.dx, originOffset.dy);
    }
  }

  /// Clips the canvas for an ink feature to be painted on the [canvas].
  ///
  /// This should be called before painting ink features with [paintFeature]
  /// that do not use [paintInkCircle].
  ///
  /// The [clipCallback] is the callback used to obtain the [Rect] used for clipping
  /// the ink effect.
  ///
  /// If [clipCallback] is null, no clipping is performed on the ink circle.
  ///
  /// The [textDirection] is used by [customBorder] if it is non-null. This
  /// allows the [customBorder]'s path to be properly defined if the path was
  /// expressed in terms of "start" and "end" instead of "left" and "right".
  ///
  /// For examples on how the function is used, see [InkSparkle].
  void _clipCanvas({
    required Canvas canvas,
    required RectCallback clipCallback,
    TextDirection? textDirection,
    ShapeBorder? customBorder,
    BorderRadius borderRadius = BorderRadius.zero,
  }) {
    final Rect rect = clipCallback();
    if (customBorder != null) {
      canvas.clipPath(
          customBorder.getOuterPath(rect, textDirection: textDirection));
    } else if (borderRadius != BorderRadius.zero) {
      canvas.clipRRect(RRect.fromRectAndCorners(
        rect,
        topLeft: borderRadius.topLeft,
        topRight: borderRadius.topRight,
        bottomLeft: borderRadius.bottomLeft,
        bottomRight: borderRadius.bottomRight,
      ));
    } else {
      canvas.clipRect(rect);
    }
  }
}

class _InkSparkleFactory extends InteractiveInkFeatureFactory {
  const _InkSparkleFactory() : turbulenceSeed = null;

  const _InkSparkleFactory.constantTurbulenceSeed() : turbulenceSeed = 1337.0;

  // TODO(clocksmith): Update this once shaders are precompiled.
  static void compileShaderIfNeccessary() {
    if (!_initCalled) {
      FragmentShaderManager.inkSparkle().then((FragmentShaderManager manager) {
        _shaderManager = manager;
      });
      _initCalled = true;
    }
  }
  static bool _initCalled = false;
  static FragmentShaderManager? _shaderManager;

  final double? turbulenceSeed;

  @override
  InteractiveInkFeature create({
    required MaterialInkController controller,
    required RenderBox referenceBox,
    required ui.Offset position,
    required ui.Color color,
    required ui.TextDirection textDirection,
    bool containedInkWell = false,
    RectCallback? rectCallback,
    BorderRadius? borderRadius,
    ShapeBorder? customBorder,
    double? radius,
    ui.VoidCallback? onRemoved,
  }) {
    return InkSparkle(
      controller: controller,
      referenceBox: referenceBox,
      position: position,
      color: color,
      textDirection: textDirection,
      containedInkWell: containedInkWell,
      rectCallback: rectCallback,
      borderRadius: borderRadius,
      customBorder: customBorder,
      radius: radius,
      onRemoved: onRemoved,
      turbulenceSeed: turbulenceSeed,
    );
  }
}

RectCallback? _getClipCallback(
  RenderBox referenceBox,
  bool containedInkWell,
  RectCallback? rectCallback,
) {
  if (rectCallback != null) {
    assert(containedInkWell);
    return rectCallback;
  }
  if (containedInkWell) {
    return () => Offset.zero & referenceBox.size;
  }
  return null;
}

double _getTargetRadius(
  RenderBox referenceBox,
  bool containedInkWell,
  RectCallback? rectCallback,
  Offset position,
) {
  final Size size = rectCallback != null ? rectCallback().size : referenceBox.size;
  final double d1 = size.bottomRight(Offset.zero).distance;
  final double d2 = (size.topRight(Offset.zero) - size.bottomLeft(Offset.zero)).distance;
  return math.max(d1, d2) / 2.0;
}

// The code below is generated by the package: fragment_shader_manager.


/// A generated class for managing [FragmentProgram] that includes a
/// pre-transpiled shader program into SPIR-V.
///
/// See:
/// - https://github.com/material-components/material-components-flutter-experimental/tree/fragment-shader-manager/fragment_shader_manager
///
/// GLSL source for this shader:
///
/// #version 320 es
///
/// precision highp float;
///
/// layout(location = 0) uniform vec4 u_color;
/// layout(location = 1) uniform float u_alpha;
/// layout(location = 2) uniform vec4 u_sparkle_color;
/// layout(location = 3) uniform float u_sparkle_alpha;
/// layout(location = 4) uniform float u_blur;
/// layout(location = 5) uniform vec2 u_center;
/// layout(location = 6) uniform float u_radius_scale;
/// layout(location = 7) uniform float u_max_radius;
/// layout(location = 8) uniform vec2 u_resolution_scale;
/// layout(location = 9) uniform vec2 u_noise_scale;
/// layout(location = 10) uniform float u_noise_phase;
/// layout(location = 11) uniform float u_turbulence_phase;
/// layout(location = 12) uniform vec2 u_circle1;
/// layout(location = 13) uniform vec2 u_circle2;
/// layout(location = 14) uniform vec2 u_circle3;
/// layout(location = 15) uniform vec2 u_rotation1;
/// layout(location = 16) uniform vec2 u_rotation2;
/// layout(location = 17) uniform vec2 u_rotation3;
///
/// layout(location = 0) out vec4 fragColor;
///
/// const float PI = 3.1415926535897932384626;
/// const float PI_ROTATE_RIGHT = PI * 0.0078125;
/// const float PI_ROTATE_LEFT = PI * -0.0078125;
/// const float ONE_THIRD = 1./3.;
/// const vec2 TURBULENCE_SCALE = vec2(0.8);
///
/// float saturate(float x) {
///   return clamp(x, 0.0, 1.0);
/// }
///
/// float triangle_noise(highp vec2 n) {
///   n = fract(n * vec2(5.3987, 5.4421));
///   n += dot(n.yx, n.xy + vec2(21.5351, 14.3137));
///   float xy = n.x * n.y;
///   return fract(xy * 95.4307) + fract(xy * 75.04961) - 1.0;
/// }
///
/// float threshold(float v, float l, float h) {
///   return step(l, v) * (1.0 - step(h, v));
/// }
///
/// mat2 rotate2d(vec2 rad){
///   return mat2(rad.x, -rad.y, rad.y, rad.x);
/// }
///
/// float soft_circle(vec2 uv, vec2 xy, float radius, float blur) {
///   float blur_half = blur * 0.5;
///   float d = distance(uv, xy);
///   return 1.0 - smoothstep(1.0 - blur_half, 1.0 + blur_half, d / radius);
/// }
///
/// float soft_ring(vec2 uv, vec2 xy, float radius, float thickness, float blur) {
///   float circle_outer = soft_circle(uv, xy, radius + thickness, blur);
///   float circle_inner = soft_circle(uv, xy, max(radius - thickness, 0.0), blur);
///   return saturate(circle_outer - circle_inner);
/// }
///
/// float circle_grid(vec2 resolution, vec2 p, float time, vec2 xy, vec2 rotation, float cell_diameter) {
///   p = rotate2d(rotation) * (xy - p) + xy;
///   p = mod(p, cell_diameter) / resolution;
///   float cell_uv = cell_diameter / resolution.y * 0.5;
///   float r = 0.65 * cell_uv;
///   return soft_circle(p, vec2(cell_uv), r, r * 50.0);
/// }
///
/// float sparkle(vec2 uv, float t) {
///   float n = triangle_noise(uv);
///   float s = threshold(n, 0.0, 0.05);
///   s += threshold(n + sin(PI * (t + 0.35)), 0.1, 0.15);
///   s += threshold(n + sin(PI * (t + 0.7)), 0.2, 0.25);
///   s += threshold(n + sin(PI * (t + 1.05)), 0.3, 0.35);
///   return saturate(s) * 0.55;
/// }
///
/// float turbulence(vec2 uv) {
///   vec2 uv_scale = uv * TURBULENCE_SCALE;
///   float g1 = circle_grid(TURBULENCE_SCALE, uv_scale, u_turbulence_phase, u_circle1, u_rotation1, 0.17);
///   float g2 = circle_grid(TURBULENCE_SCALE, uv_scale, u_turbulence_phase, u_circle2, u_rotation2, 0.2);
///   float g3 = circle_grid(TURBULENCE_SCALE, uv_scale, u_turbulence_phase, u_circle3, u_rotation3, 0.275);
///   float v = (g1 * g1 + g2 - g3) * 0.5;
///   return saturate(0.45 + 0.8 * v);
/// }
///
/// void main() {
///   vec2 p = gl_FragCoord.xy;
///   vec2 uv = p * u_resolution_scale;
///   vec2 density_uv = uv - mod(p, u_noise_scale);
///   float radius = u_max_radius * u_radius_scale;
///   float turbulence = turbulence(uv);
///   float ring = soft_ring(p, u_center, radius, 0.05 * u_max_radius, u_blur);
///   float sparkle = sparkle(density_uv, u_noise_phase) * ring * turbulence * u_sparkle_alpha;
///   float wave_alpha = soft_circle(p, u_center, radius, u_blur) * u_alpha * u_color.a;
///   vec4 wave_color = vec4(u_color.rgb * wave_alpha, wave_alpha);
///   vec4 sparkle_color = vec4(u_sparkle_color.rgb * u_sparkle_color.a, u_sparkle_color.a);
///   fragColor = mix(wave_color, sparkle_color, sparkle);
/// }
class FragmentShaderManager {
  FragmentShaderManager._();

  /// Creates an [FragmentShaderManager] with an [InkSparkle] effect.
  static Future<FragmentShaderManager> inkSparkle() async {
    final FragmentShaderManager manager = FragmentShaderManager._();
    await manager.compile();
    return manager;
  }

  /// Compiles the spir-v bytecode into a shader program.
  Future<void> compile() async {
    _program = await ui.FragmentProgram.compile(spirv: spirvByteBuffer);
  }

  /// Creates a shader with the original program and optional uniforms.
  ///
  /// A new shader must be made whenever the uniforms are updated.
  Shader shader({
    Vector4? uColor,
    double? uAlpha,
    Vector4? uSparkleColor,
    double? uSparkleAlpha,
    double? uBlur,
    Vector2? uCenter,
    double? uRadiusScale,
    double? uMaxRadius,
    Vector2? uResolutionScale,
    Vector2? uNoiseScale,
    double? uNoisePhase,
    double? uTurbulencePhase,
    Vector2? uCircle1,
    Vector2? uCircle2,
    Vector2? uCircle3,
    Vector2? uRotation1,
    Vector2? uRotation2,
    Vector2? uRotation3,
  }) {
    return _program.shader(
      floatUniforms: Float32List.fromList(<double>[
        ...uColor != null ? uColor.storage : <double>[0, 0, 0, 0],
        ...uAlpha != null ? <double>[uAlpha] : <double>[0],
        ...uSparkleColor != null ? uSparkleColor.storage : <double>[0, 0, 0, 0],
        ...uSparkleAlpha != null ? <double>[uSparkleAlpha] : <double>[0],
        ...uBlur != null ? <double>[uBlur] : <double>[0],
        ...uCenter != null ? uCenter.storage : <double>[0, 0],
        ...uRadiusScale != null ? <double>[uRadiusScale] : <double>[0],
        ...uMaxRadius != null ? <double>[uMaxRadius] : <double>[0],
        ...uResolutionScale != null ? uResolutionScale.storage : <double>[0, 0],
        ...uNoiseScale != null ? uNoiseScale.storage : <double>[0, 0],
        ...uNoisePhase != null ? <double>[uNoisePhase] : <double>[0],
        ...uTurbulencePhase != null ? <double>[uTurbulencePhase] : <double>[0],
        ...uCircle1 != null ? uCircle1.storage : <double>[0, 0],
        ...uCircle2 != null ? uCircle2.storage : <double>[0, 0],
        ...uCircle3 != null ? uCircle3.storage : <double>[0, 0],
        ...uRotation1 != null ? uRotation1.storage : <double>[0, 0],
        ...uRotation2 != null ? uRotation2.storage : <double>[0, 0],
        ...uRotation3 != null ? uRotation3.storage : <double>[0, 0],
      ]),
    );
  }

  late ui.FragmentProgram _program;

  /// Direct access to the [ui.FragmentProgram] that this class manages.
  ///
  /// In general, this is not needed, but may be useful for debugging or edge cases.
  ui.FragmentProgram get program => _program;

  /// Direct access to the the SPIR-V bytecode that was used to generate this class.
  ///
  /// In general, this is not needed, but may be useful for debugging or edge cases.
  ///
  /// Words in SPIR-V are 32 bits. Every 4 elements in this list represents 1
  /// SPIR-V word. See https://www.khronos.org/registry/SPIR-V/.
  ByteBuffer get spirvByteBuffer => Uint8List.fromList(_spirvByteList).buffer;
}

// TODO(clocksmith): Move spirv to binary asset.
// https://github.com/flutter/flutter/issues/99783

/// 11900 bytes
const List<int> _spirvByteList = <int>[
  0x03,0x02,0x23,0x07,
  0x00,0x00,0x01,0x00,
  0x0A,0x00,0x0D,0x00,
  0xCE,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x11,0x00,0x02,0x00,
  0x01,0x00,0x00,0x00,
  0x0B,0x00,0x06,0x00,
  0x01,0x00,0x00,0x00,
  0x47,0x4C,0x53,0x4C,
  0x2E,0x73,0x74,0x64,
  0x2E,0x34,0x35,0x30,
  0x00,0x00,0x00,0x00,
  0x0E,0x00,0x03,0x00,
  0x00,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x0F,0x00,0x07,0x00,
  0x04,0x00,0x00,0x00,
  0x04,0x00,0x00,0x00,
  0x6D,0x61,0x69,0x6E,
  0x00,0x00,0x00,0x00,
  0x62,0x01,0x00,0x00,
  0xC5,0x01,0x00,0x00,
  0x10,0x00,0x03,0x00,
  0x04,0x00,0x00,0x00,
  0x08,0x00,0x00,0x00,
  0x03,0x00,0x03,0x00,
  0x01,0x00,0x00,0x00,
  0x40,0x01,0x00,0x00,
  0x04,0x00,0x0A,0x00,
  0x47,0x4C,0x5F,0x47,
  0x4F,0x4F,0x47,0x4C,
  0x45,0x5F,0x63,0x70,
  0x70,0x5F,0x73,0x74,
  0x79,0x6C,0x65,0x5F,
  0x6C,0x69,0x6E,0x65,
  0x5F,0x64,0x69,0x72,
  0x65,0x63,0x74,0x69,
  0x76,0x65,0x00,0x00,
  0x04,0x00,0x08,0x00,
  0x47,0x4C,0x5F,0x47,
  0x4F,0x4F,0x47,0x4C,
  0x45,0x5F,0x69,0x6E,
  0x63,0x6C,0x75,0x64,
  0x65,0x5F,0x64,0x69,
  0x72,0x65,0x63,0x74,
  0x69,0x76,0x65,0x00,
  0x05,0x00,0x04,0x00,
  0x04,0x00,0x00,0x00,
  0x6D,0x61,0x69,0x6E,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x0A,0x00,0x00,0x00,
  0x73,0x61,0x74,0x75,
  0x72,0x61,0x74,0x65,
  0x28,0x66,0x31,0x3B,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x09,0x00,0x00,0x00,
  0x78,0x00,0x00,0x00,
  0x05,0x00,0x07,0x00,
  0x10,0x00,0x00,0x00,
  0x74,0x72,0x69,0x61,
  0x6E,0x67,0x6C,0x65,
  0x5F,0x6E,0x6F,0x69,
  0x73,0x65,0x28,0x76,
  0x66,0x32,0x3B,0x00,
  0x05,0x00,0x03,0x00,
  0x0F,0x00,0x00,0x00,
  0x6E,0x00,0x00,0x00,
  0x05,0x00,0x07,0x00,
  0x16,0x00,0x00,0x00,
  0x74,0x68,0x72,0x65,
  0x73,0x68,0x6F,0x6C,
  0x64,0x28,0x66,0x31,
  0x3B,0x66,0x31,0x3B,
  0x66,0x31,0x3B,0x00,
  0x05,0x00,0x03,0x00,
  0x13,0x00,0x00,0x00,
  0x76,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x14,0x00,0x00,0x00,
  0x6C,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x15,0x00,0x00,0x00,
  0x68,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x1B,0x00,0x00,0x00,
  0x72,0x6F,0x74,0x61,
  0x74,0x65,0x32,0x64,
  0x28,0x76,0x66,0x32,
  0x3B,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x1A,0x00,0x00,0x00,
  0x72,0x61,0x64,0x00,
  0x05,0x00,0x09,0x00,
  0x22,0x00,0x00,0x00,
  0x73,0x6F,0x66,0x74,
  0x5F,0x63,0x69,0x72,
  0x63,0x6C,0x65,0x28,
  0x76,0x66,0x32,0x3B,
  0x76,0x66,0x32,0x3B,
  0x66,0x31,0x3B,0x66,
  0x31,0x3B,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x1E,0x00,0x00,0x00,
  0x75,0x76,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x1F,0x00,0x00,0x00,
  0x78,0x79,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x20,0x00,0x00,0x00,
  0x72,0x61,0x64,0x69,
  0x75,0x73,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x21,0x00,0x00,0x00,
  0x62,0x6C,0x75,0x72,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x09,0x00,
  0x2A,0x00,0x00,0x00,
  0x73,0x6F,0x66,0x74,
  0x5F,0x72,0x69,0x6E,
  0x67,0x28,0x76,0x66,
  0x32,0x3B,0x76,0x66,
  0x32,0x3B,0x66,0x31,
  0x3B,0x66,0x31,0x3B,
  0x66,0x31,0x3B,0x00,
  0x05,0x00,0x03,0x00,
  0x25,0x00,0x00,0x00,
  0x75,0x76,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x26,0x00,0x00,0x00,
  0x78,0x79,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x27,0x00,0x00,0x00,
  0x72,0x61,0x64,0x69,
  0x75,0x73,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x28,0x00,0x00,0x00,
  0x74,0x68,0x69,0x63,
  0x6B,0x6E,0x65,0x73,
  0x73,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x29,0x00,0x00,0x00,
  0x62,0x6C,0x75,0x72,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x0B,0x00,
  0x33,0x00,0x00,0x00,
  0x63,0x69,0x72,0x63,
  0x6C,0x65,0x5F,0x67,
  0x72,0x69,0x64,0x28,
  0x76,0x66,0x32,0x3B,
  0x76,0x66,0x32,0x3B,
  0x66,0x31,0x3B,0x76,
  0x66,0x32,0x3B,0x76,
  0x66,0x32,0x3B,0x66,
  0x31,0x3B,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x2D,0x00,0x00,0x00,
  0x72,0x65,0x73,0x6F,
  0x6C,0x75,0x74,0x69,
  0x6F,0x6E,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x2E,0x00,0x00,0x00,
  0x70,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x2F,0x00,0x00,0x00,
  0x74,0x69,0x6D,0x65,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x30,0x00,0x00,0x00,
  0x78,0x79,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x31,0x00,0x00,0x00,
  0x72,0x6F,0x74,0x61,
  0x74,0x69,0x6F,0x6E,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x32,0x00,0x00,0x00,
  0x63,0x65,0x6C,0x6C,
  0x5F,0x64,0x69,0x61,
  0x6D,0x65,0x74,0x65,
  0x72,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x38,0x00,0x00,0x00,
  0x73,0x70,0x61,0x72,
  0x6B,0x6C,0x65,0x28,
  0x76,0x66,0x32,0x3B,
  0x66,0x31,0x3B,0x00,
  0x05,0x00,0x03,0x00,
  0x36,0x00,0x00,0x00,
  0x75,0x76,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x37,0x00,0x00,0x00,
  0x74,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x3B,0x00,0x00,0x00,
  0x74,0x75,0x72,0x62,
  0x75,0x6C,0x65,0x6E,
  0x63,0x65,0x28,0x76,
  0x66,0x32,0x3B,0x00,
  0x05,0x00,0x03,0x00,
  0x3A,0x00,0x00,0x00,
  0x75,0x76,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x54,0x00,0x00,0x00,
  0x78,0x79,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x81,0x00,0x00,0x00,
  0x62,0x6C,0x75,0x72,
  0x5F,0x68,0x61,0x6C,
  0x66,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x85,0x00,0x00,0x00,
  0x64,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x94,0x00,0x00,0x00,
  0x63,0x69,0x72,0x63,
  0x6C,0x65,0x5F,0x6F,
  0x75,0x74,0x65,0x72,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x98,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x9A,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x9C,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x9D,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0xA0,0x00,0x00,0x00,
  0x63,0x69,0x72,0x63,
  0x6C,0x65,0x5F,0x69,
  0x6E,0x6E,0x65,0x72,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xA5,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xA7,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xA9,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xAA,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xB0,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xB4,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xC3,0x00,0x00,0x00,
  0x63,0x65,0x6C,0x6C,
  0x5F,0x75,0x76,0x00,
  0x05,0x00,0x03,0x00,
  0xC9,0x00,0x00,0x00,
  0x72,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xD2,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xD4,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xD5,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xD7,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0xDB,0x00,0x00,0x00,
  0x6E,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xDC,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0xDF,0x00,0x00,0x00,
  0x73,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xE1,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xE3,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xE4,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xF0,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xF1,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xF2,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xFF,0x00,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x00,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x01,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x0D,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x0E,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x0F,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x13,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x1A,0x01,0x00,0x00,
  0x75,0x76,0x5F,0x73,
  0x63,0x61,0x6C,0x65,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x1F,0x01,0x00,0x00,
  0x67,0x31,0x00,0x00,
  0x05,0x00,0x07,0x00,
  0x21,0x01,0x00,0x00,
  0x75,0x5F,0x74,0x75,
  0x72,0x62,0x75,0x6C,
  0x65,0x6E,0x63,0x65,
  0x5F,0x70,0x68,0x61,
  0x73,0x65,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x23,0x01,0x00,0x00,
  0x75,0x5F,0x63,0x69,
  0x72,0x63,0x6C,0x65,
  0x31,0x00,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x24,0x01,0x00,0x00,
  0x75,0x5F,0x72,0x6F,
  0x74,0x61,0x74,0x69,
  0x6F,0x6E,0x31,0x00,
  0x05,0x00,0x04,0x00,
  0x26,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x27,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x29,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x2B,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x2D,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x2F,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x31,0x01,0x00,0x00,
  0x67,0x32,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x32,0x01,0x00,0x00,
  0x75,0x5F,0x63,0x69,
  0x72,0x63,0x6C,0x65,
  0x32,0x00,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x33,0x01,0x00,0x00,
  0x75,0x5F,0x72,0x6F,
  0x74,0x61,0x74,0x69,
  0x6F,0x6E,0x32,0x00,
  0x05,0x00,0x04,0x00,
  0x34,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x35,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x37,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x39,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x3B,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x3D,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x3F,0x01,0x00,0x00,
  0x67,0x33,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x40,0x01,0x00,0x00,
  0x75,0x5F,0x63,0x69,
  0x72,0x63,0x6C,0x65,
  0x33,0x00,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x41,0x01,0x00,0x00,
  0x75,0x5F,0x72,0x6F,
  0x74,0x61,0x74,0x69,
  0x6F,0x6E,0x33,0x00,
  0x05,0x00,0x04,0x00,
  0x43,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x44,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x46,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x48,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x4A,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x4C,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x4E,0x01,0x00,0x00,
  0x76,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x5B,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x5F,0x01,0x00,0x00,
  0x70,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x62,0x01,0x00,0x00,
  0x67,0x6C,0x5F,0x46,
  0x72,0x61,0x67,0x43,
  0x6F,0x6F,0x72,0x64,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x03,0x00,
  0x65,0x01,0x00,0x00,
  0x75,0x76,0x00,0x00,
  0x05,0x00,0x07,0x00,
  0x67,0x01,0x00,0x00,
  0x75,0x5F,0x72,0x65,
  0x73,0x6F,0x6C,0x75,
  0x74,0x69,0x6F,0x6E,
  0x5F,0x73,0x63,0x61,
  0x6C,0x65,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x6A,0x01,0x00,0x00,
  0x64,0x65,0x6E,0x73,
  0x69,0x74,0x79,0x5F,
  0x75,0x76,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x6D,0x01,0x00,0x00,
  0x75,0x5F,0x6E,0x6F,
  0x69,0x73,0x65,0x5F,
  0x73,0x63,0x61,0x6C,
  0x65,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x71,0x01,0x00,0x00,
  0x72,0x61,0x64,0x69,
  0x75,0x73,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x72,0x01,0x00,0x00,
  0x75,0x5F,0x6D,0x61,
  0x78,0x5F,0x72,0x61,
  0x64,0x69,0x75,0x73,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x74,0x01,0x00,0x00,
  0x75,0x5F,0x72,0x61,
  0x64,0x69,0x75,0x73,
  0x5F,0x73,0x63,0x61,
  0x6C,0x65,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x77,0x01,0x00,0x00,
  0x74,0x75,0x72,0x62,
  0x75,0x6C,0x65,0x6E,
  0x63,0x65,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x78,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x7B,0x01,0x00,0x00,
  0x72,0x69,0x6E,0x67,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x05,0x00,
  0x7C,0x01,0x00,0x00,
  0x75,0x5F,0x63,0x65,
  0x6E,0x74,0x65,0x72,
  0x00,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x7F,0x01,0x00,0x00,
  0x75,0x5F,0x62,0x6C,
  0x75,0x72,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x80,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x82,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x84,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x86,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x87,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x8A,0x01,0x00,0x00,
  0x73,0x70,0x61,0x72,
  0x6B,0x6C,0x65,0x00,
  0x05,0x00,0x06,0x00,
  0x8B,0x01,0x00,0x00,
  0x75,0x5F,0x6E,0x6F,
  0x69,0x73,0x65,0x5F,
  0x70,0x68,0x61,0x73,
  0x65,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x8C,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x8E,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0x95,0x01,0x00,0x00,
  0x75,0x5F,0x73,0x70,
  0x61,0x72,0x6B,0x6C,
  0x65,0x5F,0x61,0x6C,
  0x70,0x68,0x61,0x00,
  0x05,0x00,0x05,0x00,
  0x98,0x01,0x00,0x00,
  0x77,0x61,0x76,0x65,
  0x5F,0x61,0x6C,0x70,
  0x68,0x61,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x99,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x9B,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x9D,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0x9F,0x01,0x00,0x00,
  0x70,0x61,0x72,0x61,
  0x6D,0x00,0x00,0x00,
  0x05,0x00,0x04,0x00,
  0xA2,0x01,0x00,0x00,
  0x75,0x5F,0x61,0x6C,
  0x70,0x68,0x61,0x00,
  0x05,0x00,0x04,0x00,
  0xA6,0x01,0x00,0x00,
  0x75,0x5F,0x63,0x6F,
  0x6C,0x6F,0x72,0x00,
  0x05,0x00,0x05,0x00,
  0xAC,0x01,0x00,0x00,
  0x77,0x61,0x76,0x65,
  0x5F,0x63,0x6F,0x6C,
  0x6F,0x72,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0xB7,0x01,0x00,0x00,
  0x73,0x70,0x61,0x72,
  0x6B,0x6C,0x65,0x5F,
  0x63,0x6F,0x6C,0x6F,
  0x72,0x00,0x00,0x00,
  0x05,0x00,0x06,0x00,
  0xB8,0x01,0x00,0x00,
  0x75,0x5F,0x73,0x70,
  0x61,0x72,0x6B,0x6C,
  0x65,0x5F,0x63,0x6F,
  0x6C,0x6F,0x72,0x00,
  0x05,0x00,0x05,0x00,
  0xC5,0x01,0x00,0x00,
  0x66,0x72,0x61,0x67,
  0x43,0x6F,0x6C,0x6F,
  0x72,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x21,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x0B,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x23,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x0C,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x24,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x32,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x33,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x10,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x40,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x0E,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x41,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x11,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x62,0x01,0x00,0x00,
  0x0B,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x67,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x08,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x6D,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x09,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x72,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x74,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x7C,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x05,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x7F,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x04,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x8B,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0x95,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x03,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0xA2,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0xA6,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0xB8,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x47,0x00,0x04,0x00,
  0xC5,0x01,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x13,0x00,0x02,0x00,
  0x02,0x00,0x00,0x00,
  0x21,0x00,0x03,0x00,
  0x03,0x00,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x16,0x00,0x03,0x00,
  0x06,0x00,0x00,0x00,
  0x20,0x00,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x21,0x00,0x04,0x00,
  0x08,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x17,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x0C,0x00,0x00,0x00,
  0x21,0x00,0x04,0x00,
  0x0E,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x21,0x00,0x06,0x00,
  0x12,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x18,0x00,0x04,0x00,
  0x18,0x00,0x00,0x00,
  0x0C,0x00,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x21,0x00,0x04,0x00,
  0x19,0x00,0x00,0x00,
  0x18,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x21,0x00,0x07,0x00,
  0x1D,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x21,0x00,0x08,0x00,
  0x24,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x21,0x00,0x09,0x00,
  0x2C,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x21,0x00,0x05,0x00,
  0x35,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x3E,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x3F,0x00,0x00,0x00,
  0x00,0x00,0x80,0x3F,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x44,0x00,0x00,0x00,
  0x27,0xC2,0xAC,0x40,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x45,0x00,0x00,0x00,
  0xAF,0x25,0xAE,0x40,
  0x2C,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x46,0x00,0x00,0x00,
  0x44,0x00,0x00,0x00,
  0x45,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x4C,0x00,0x00,0x00,
  0xE3,0x47,0xAC,0x41,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x4D,0x00,0x00,0x00,
  0xEA,0x04,0x65,0x41,
  0x2C,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x4E,0x00,0x00,0x00,
  0x4C,0x00,0x00,0x00,
  0x4D,0x00,0x00,0x00,
  0x15,0x00,0x04,0x00,
  0x55,0x00,0x00,0x00,
  0x20,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x55,0x00,0x00,0x00,
  0x56,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x55,0x00,0x00,0x00,
  0x59,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x5E,0x00,0x00,0x00,
  0x85,0xDC,0xBE,0x42,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x62,0x00,0x00,0x00,
  0x66,0x19,0x96,0x42,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x83,0x00,0x00,0x00,
  0x00,0x00,0x00,0x3F,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xCA,0x00,0x00,0x00,
  0x66,0x66,0x26,0x3F,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xD0,0x00,0x00,0x00,
  0x00,0x00,0x48,0x42,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xE0,0x00,0x00,0x00,
  0xCD,0xCC,0x4C,0x3D,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xE7,0x00,0x00,0x00,
  0xDB,0x0F,0x49,0x40,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xE9,0x00,0x00,0x00,
  0x33,0x33,0xB3,0x3E,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xEE,0x00,0x00,0x00,
  0xCD,0xCC,0xCC,0x3D,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xEF,0x00,0x00,0x00,
  0x9A,0x99,0x19,0x3E,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xF8,0x00,0x00,0x00,
  0x33,0x33,0x33,0x3F,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xFD,0x00,0x00,0x00,
  0xCD,0xCC,0x4C,0x3E,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xFE,0x00,0x00,0x00,
  0x00,0x00,0x80,0x3E,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x07,0x01,0x00,0x00,
  0x66,0x66,0x86,0x3F,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x0C,0x01,0x00,0x00,
  0x9A,0x99,0x99,0x3E,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x16,0x01,0x00,0x00,
  0xCD,0xCC,0x0C,0x3F,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x1C,0x01,0x00,0x00,
  0xCD,0xCC,0x4C,0x3F,
  0x2C,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x1D,0x01,0x00,0x00,
  0x1C,0x01,0x00,0x00,
  0x1C,0x01,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0x21,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x0C,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x23,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x24,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x25,0x01,0x00,0x00,
  0x7B,0x14,0x2E,0x3E,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x32,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x33,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x40,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x41,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x42,0x01,0x00,0x00,
  0xCD,0xCC,0x8C,0x3E,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x57,0x01,0x00,0x00,
  0x66,0x66,0xE6,0x3E,
  0x17,0x00,0x04,0x00,
  0x60,0x01,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x04,0x00,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0x61,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x60,0x01,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x61,0x01,0x00,0x00,
  0x62,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x67,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x6D,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0x72,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0x74,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x22,0x01,0x00,0x00,
  0x7C,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0x7F,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0x8B,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0x95,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x20,0x01,0x00,0x00,
  0xA2,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0xA5,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x60,0x01,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0xA5,0x01,0x00,0x00,
  0xA6,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x55,0x00,0x00,0x00,
  0xA7,0x01,0x00,0x00,
  0x03,0x00,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0xAB,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x60,0x01,0x00,0x00,
  0x17,0x00,0x04,0x00,
  0xAD,0x01,0x00,0x00,
  0x06,0x00,0x00,0x00,
  0x03,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0xA5,0x01,0x00,0x00,
  0xB8,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x20,0x00,0x04,0x00,
  0xC4,0x01,0x00,0x00,
  0x03,0x00,0x00,0x00,
  0x60,0x01,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0xC4,0x01,0x00,0x00,
  0xC5,0x01,0x00,0x00,
  0x03,0x00,0x00,0x00,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xCB,0x01,0x00,0x00,
  0xDB,0x0F,0xC9,0x3C,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xCC,0x01,0x00,0x00,
  0xDB,0x0F,0xC9,0xBC,
  0x2B,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xCD,0x01,0x00,0x00,
  0xAB,0xAA,0xAA,0x3E,
  0x36,0x00,0x05,0x00,
  0x02,0x00,0x00,0x00,
  0x04,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x03,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x05,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x5F,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x65,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x6A,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x71,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x77,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x78,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x7B,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x80,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x82,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x84,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x86,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x87,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x8A,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x8C,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x8E,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x98,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x99,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x9B,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x9D,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x9F,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0xAB,0x01,0x00,0x00,
  0xAC,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0xAB,0x01,0x00,0x00,
  0xB7,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x60,0x01,0x00,0x00,
  0x63,0x01,0x00,0x00,
  0x62,0x01,0x00,0x00,
  0x4F,0x00,0x07,0x00,
  0x0C,0x00,0x00,0x00,
  0x64,0x01,0x00,0x00,
  0x63,0x01,0x00,0x00,
  0x63,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x5F,0x01,0x00,0x00,
  0x64,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x66,0x01,0x00,0x00,
  0x5F,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x68,0x01,0x00,0x00,
  0x67,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x69,0x01,0x00,0x00,
  0x66,0x01,0x00,0x00,
  0x68,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x65,0x01,0x00,0x00,
  0x69,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x6B,0x01,0x00,0x00,
  0x65,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x6C,0x01,0x00,0x00,
  0x5F,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x6E,0x01,0x00,0x00,
  0x6D,0x01,0x00,0x00,
  0x8D,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x6F,0x01,0x00,0x00,
  0x6C,0x01,0x00,0x00,
  0x6E,0x01,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x70,0x01,0x00,0x00,
  0x6B,0x01,0x00,0x00,
  0x6F,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x6A,0x01,0x00,0x00,
  0x70,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x73,0x01,0x00,0x00,
  0x72,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x75,0x01,0x00,0x00,
  0x74,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x76,0x01,0x00,0x00,
  0x73,0x01,0x00,0x00,
  0x75,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x71,0x01,0x00,0x00,
  0x76,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x79,0x01,0x00,0x00,
  0x65,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x78,0x01,0x00,0x00,
  0x79,0x01,0x00,0x00,
  0x39,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x7A,0x01,0x00,0x00,
  0x3B,0x00,0x00,0x00,
  0x78,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x77,0x01,0x00,0x00,
  0x7A,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x7D,0x01,0x00,0x00,
  0x72,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x7E,0x01,0x00,0x00,
  0xE0,0x00,0x00,0x00,
  0x7D,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x81,0x01,0x00,0x00,
  0x5F,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x80,0x01,0x00,0x00,
  0x81,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x83,0x01,0x00,0x00,
  0x7C,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x82,0x01,0x00,0x00,
  0x83,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x85,0x01,0x00,0x00,
  0x71,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x84,0x01,0x00,0x00,
  0x85,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x86,0x01,0x00,0x00,
  0x7E,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x88,0x01,0x00,0x00,
  0x7F,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x87,0x01,0x00,0x00,
  0x88,0x01,0x00,0x00,
  0x39,0x00,0x09,0x00,
  0x06,0x00,0x00,0x00,
  0x89,0x01,0x00,0x00,
  0x2A,0x00,0x00,0x00,
  0x80,0x01,0x00,0x00,
  0x82,0x01,0x00,0x00,
  0x84,0x01,0x00,0x00,
  0x86,0x01,0x00,0x00,
  0x87,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x7B,0x01,0x00,0x00,
  0x89,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x8D,0x01,0x00,0x00,
  0x6A,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x8C,0x01,0x00,0x00,
  0x8D,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x8F,0x01,0x00,0x00,
  0x8B,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x8E,0x01,0x00,0x00,
  0x8F,0x01,0x00,0x00,
  0x39,0x00,0x06,0x00,
  0x06,0x00,0x00,0x00,
  0x90,0x01,0x00,0x00,
  0x38,0x00,0x00,0x00,
  0x8C,0x01,0x00,0x00,
  0x8E,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x91,0x01,0x00,0x00,
  0x7B,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x92,0x01,0x00,0x00,
  0x90,0x01,0x00,0x00,
  0x91,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x93,0x01,0x00,0x00,
  0x77,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x94,0x01,0x00,0x00,
  0x92,0x01,0x00,0x00,
  0x93,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x96,0x01,0x00,0x00,
  0x95,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x97,0x01,0x00,0x00,
  0x94,0x01,0x00,0x00,
  0x96,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x8A,0x01,0x00,0x00,
  0x97,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x9A,0x01,0x00,0x00,
  0x5F,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x99,0x01,0x00,0x00,
  0x9A,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x9C,0x01,0x00,0x00,
  0x7C,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x9B,0x01,0x00,0x00,
  0x9C,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x9E,0x01,0x00,0x00,
  0x71,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x9D,0x01,0x00,0x00,
  0x9E,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xA0,0x01,0x00,0x00,
  0x7F,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x9F,0x01,0x00,0x00,
  0xA0,0x01,0x00,0x00,
  0x39,0x00,0x08,0x00,
  0x06,0x00,0x00,0x00,
  0xA1,0x01,0x00,0x00,
  0x22,0x00,0x00,0x00,
  0x99,0x01,0x00,0x00,
  0x9B,0x01,0x00,0x00,
  0x9D,0x01,0x00,0x00,
  0x9F,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xA3,0x01,0x00,0x00,
  0xA2,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xA4,0x01,0x00,0x00,
  0xA1,0x01,0x00,0x00,
  0xA3,0x01,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x20,0x01,0x00,0x00,
  0xA8,0x01,0x00,0x00,
  0xA6,0x01,0x00,0x00,
  0xA7,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xA9,0x01,0x00,0x00,
  0xA8,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xAA,0x01,0x00,0x00,
  0xA4,0x01,0x00,0x00,
  0xA9,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x98,0x01,0x00,0x00,
  0xAA,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x60,0x01,0x00,0x00,
  0xAE,0x01,0x00,0x00,
  0xA6,0x01,0x00,0x00,
  0x4F,0x00,0x08,0x00,
  0xAD,0x01,0x00,0x00,
  0xAF,0x01,0x00,0x00,
  0xAE,0x01,0x00,0x00,
  0xAE,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xB0,0x01,0x00,0x00,
  0x98,0x01,0x00,0x00,
  0x8E,0x00,0x05,0x00,
  0xAD,0x01,0x00,0x00,
  0xB1,0x01,0x00,0x00,
  0xAF,0x01,0x00,0x00,
  0xB0,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xB2,0x01,0x00,0x00,
  0x98,0x01,0x00,0x00,
  0x51,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xB3,0x01,0x00,0x00,
  0xB1,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x51,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xB4,0x01,0x00,0x00,
  0xB1,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x51,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xB5,0x01,0x00,0x00,
  0xB1,0x01,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x50,0x00,0x07,0x00,
  0x60,0x01,0x00,0x00,
  0xB6,0x01,0x00,0x00,
  0xB3,0x01,0x00,0x00,
  0xB4,0x01,0x00,0x00,
  0xB5,0x01,0x00,0x00,
  0xB2,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xAC,0x01,0x00,0x00,
  0xB6,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x60,0x01,0x00,0x00,
  0xB9,0x01,0x00,0x00,
  0xB8,0x01,0x00,0x00,
  0x4F,0x00,0x08,0x00,
  0xAD,0x01,0x00,0x00,
  0xBA,0x01,0x00,0x00,
  0xB9,0x01,0x00,0x00,
  0xB9,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x20,0x01,0x00,0x00,
  0xBB,0x01,0x00,0x00,
  0xB8,0x01,0x00,0x00,
  0xA7,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xBC,0x01,0x00,0x00,
  0xBB,0x01,0x00,0x00,
  0x8E,0x00,0x05,0x00,
  0xAD,0x01,0x00,0x00,
  0xBD,0x01,0x00,0x00,
  0xBA,0x01,0x00,0x00,
  0xBC,0x01,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x20,0x01,0x00,0x00,
  0xBE,0x01,0x00,0x00,
  0xB8,0x01,0x00,0x00,
  0xA7,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xBF,0x01,0x00,0x00,
  0xBE,0x01,0x00,0x00,
  0x51,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xC0,0x01,0x00,0x00,
  0xBD,0x01,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x51,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xC1,0x01,0x00,0x00,
  0xBD,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x51,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xC2,0x01,0x00,0x00,
  0xBD,0x01,0x00,0x00,
  0x02,0x00,0x00,0x00,
  0x50,0x00,0x07,0x00,
  0x60,0x01,0x00,0x00,
  0xC3,0x01,0x00,0x00,
  0xC0,0x01,0x00,0x00,
  0xC1,0x01,0x00,0x00,
  0xC2,0x01,0x00,0x00,
  0xBF,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xB7,0x01,0x00,0x00,
  0xC3,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x60,0x01,0x00,0x00,
  0xC6,0x01,0x00,0x00,
  0xAC,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x60,0x01,0x00,0x00,
  0xC7,0x01,0x00,0x00,
  0xB7,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xC8,0x01,0x00,0x00,
  0x8A,0x01,0x00,0x00,
  0x50,0x00,0x07,0x00,
  0x60,0x01,0x00,0x00,
  0xC9,0x01,0x00,0x00,
  0xC8,0x01,0x00,0x00,
  0xC8,0x01,0x00,0x00,
  0xC8,0x01,0x00,0x00,
  0xC8,0x01,0x00,0x00,
  0x0C,0x00,0x08,0x00,
  0x60,0x01,0x00,0x00,
  0xCA,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x2E,0x00,0x00,0x00,
  0xC6,0x01,0x00,0x00,
  0xC7,0x01,0x00,0x00,
  0xC9,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xC5,0x01,0x00,0x00,
  0xCA,0x01,0x00,0x00,
  0xFD,0x00,0x01,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x08,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x09,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x0B,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x3D,0x00,0x00,0x00,
  0x09,0x00,0x00,0x00,
  0x0C,0x00,0x08,0x00,
  0x06,0x00,0x00,0x00,
  0x40,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x2B,0x00,0x00,0x00,
  0x3D,0x00,0x00,0x00,
  0x3E,0x00,0x00,0x00,
  0x3F,0x00,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0x40,0x00,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x10,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x0E,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x11,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x54,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x43,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x47,0x00,0x00,0x00,
  0x43,0x00,0x00,0x00,
  0x46,0x00,0x00,0x00,
  0x0C,0x00,0x06,0x00,
  0x0C,0x00,0x00,0x00,
  0x48,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0x47,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x0F,0x00,0x00,0x00,
  0x48,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x49,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x4F,0x00,0x07,0x00,
  0x0C,0x00,0x00,0x00,
  0x4A,0x00,0x00,0x00,
  0x49,0x00,0x00,0x00,
  0x49,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x4B,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x4F,0x00,0x00,0x00,
  0x4B,0x00,0x00,0x00,
  0x4E,0x00,0x00,0x00,
  0x94,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x50,0x00,0x00,0x00,
  0x4A,0x00,0x00,0x00,
  0x4F,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x51,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x50,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x52,0x00,0x00,0x00,
  0x50,0x00,0x00,0x00,
  0x50,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x53,0x00,0x00,0x00,
  0x51,0x00,0x00,0x00,
  0x52,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x0F,0x00,0x00,0x00,
  0x53,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x07,0x00,0x00,0x00,
  0x57,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x56,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x58,0x00,0x00,0x00,
  0x57,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x07,0x00,0x00,0x00,
  0x5A,0x00,0x00,0x00,
  0x0F,0x00,0x00,0x00,
  0x59,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x5B,0x00,0x00,0x00,
  0x5A,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x5C,0x00,0x00,0x00,
  0x58,0x00,0x00,0x00,
  0x5B,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x54,0x00,0x00,0x00,
  0x5C,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x5D,0x00,0x00,0x00,
  0x54,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x5F,0x00,0x00,0x00,
  0x5D,0x00,0x00,0x00,
  0x5E,0x00,0x00,0x00,
  0x0C,0x00,0x06,0x00,
  0x06,0x00,0x00,0x00,
  0x60,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0x5F,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x61,0x00,0x00,0x00,
  0x54,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x63,0x00,0x00,0x00,
  0x61,0x00,0x00,0x00,
  0x62,0x00,0x00,0x00,
  0x0C,0x00,0x06,0x00,
  0x06,0x00,0x00,0x00,
  0x64,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0x63,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x65,0x00,0x00,0x00,
  0x60,0x00,0x00,0x00,
  0x64,0x00,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x66,0x00,0x00,0x00,
  0x65,0x00,0x00,0x00,
  0x3F,0x00,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0x66,0x00,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x16,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x12,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x13,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x14,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x15,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x17,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x69,0x00,0x00,0x00,
  0x14,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x6A,0x00,0x00,0x00,
  0x13,0x00,0x00,0x00,
  0x0C,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0x6B,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x30,0x00,0x00,0x00,
  0x69,0x00,0x00,0x00,
  0x6A,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x6C,0x00,0x00,0x00,
  0x15,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x6D,0x00,0x00,0x00,
  0x13,0x00,0x00,0x00,
  0x0C,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0x6E,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x30,0x00,0x00,0x00,
  0x6C,0x00,0x00,0x00,
  0x6D,0x00,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x6F,0x00,0x00,0x00,
  0x3F,0x00,0x00,0x00,
  0x6E,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x70,0x00,0x00,0x00,
  0x6B,0x00,0x00,0x00,
  0x6F,0x00,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0x70,0x00,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x18,0x00,0x00,0x00,
  0x1B,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x19,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x1A,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x1C,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x07,0x00,0x00,0x00,
  0x73,0x00,0x00,0x00,
  0x1A,0x00,0x00,0x00,
  0x56,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x74,0x00,0x00,0x00,
  0x73,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x07,0x00,0x00,0x00,
  0x75,0x00,0x00,0x00,
  0x1A,0x00,0x00,0x00,
  0x59,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x76,0x00,0x00,0x00,
  0x75,0x00,0x00,0x00,
  0x7F,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x77,0x00,0x00,0x00,
  0x76,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x07,0x00,0x00,0x00,
  0x78,0x00,0x00,0x00,
  0x1A,0x00,0x00,0x00,
  0x59,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x79,0x00,0x00,0x00,
  0x78,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x07,0x00,0x00,0x00,
  0x7A,0x00,0x00,0x00,
  0x1A,0x00,0x00,0x00,
  0x56,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x7B,0x00,0x00,0x00,
  0x7A,0x00,0x00,0x00,
  0x50,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x7C,0x00,0x00,0x00,
  0x74,0x00,0x00,0x00,
  0x77,0x00,0x00,0x00,
  0x50,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x7D,0x00,0x00,0x00,
  0x79,0x00,0x00,0x00,
  0x7B,0x00,0x00,0x00,
  0x50,0x00,0x05,0x00,
  0x18,0x00,0x00,0x00,
  0x7E,0x00,0x00,0x00,
  0x7C,0x00,0x00,0x00,
  0x7D,0x00,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0x7E,0x00,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x22,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x1D,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x1F,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x20,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x21,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x23,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x81,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x85,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x82,0x00,0x00,0x00,
  0x21,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x84,0x00,0x00,0x00,
  0x82,0x00,0x00,0x00,
  0x83,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x81,0x00,0x00,0x00,
  0x84,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x86,0x00,0x00,0x00,
  0x1E,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x87,0x00,0x00,0x00,
  0x1F,0x00,0x00,0x00,
  0x0C,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0x88,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x43,0x00,0x00,0x00,
  0x86,0x00,0x00,0x00,
  0x87,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x85,0x00,0x00,0x00,
  0x88,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x89,0x00,0x00,0x00,
  0x81,0x00,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x8A,0x00,0x00,0x00,
  0x3F,0x00,0x00,0x00,
  0x89,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x8B,0x00,0x00,0x00,
  0x81,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x8C,0x00,0x00,0x00,
  0x3F,0x00,0x00,0x00,
  0x8B,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x8D,0x00,0x00,0x00,
  0x85,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x8E,0x00,0x00,0x00,
  0x20,0x00,0x00,0x00,
  0x88,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x8F,0x00,0x00,0x00,
  0x8D,0x00,0x00,0x00,
  0x8E,0x00,0x00,0x00,
  0x0C,0x00,0x08,0x00,
  0x06,0x00,0x00,0x00,
  0x90,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x31,0x00,0x00,0x00,
  0x8A,0x00,0x00,0x00,
  0x8C,0x00,0x00,0x00,
  0x8F,0x00,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x91,0x00,0x00,0x00,
  0x3F,0x00,0x00,0x00,
  0x90,0x00,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0x91,0x00,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x2A,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x24,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x25,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x26,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x27,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x28,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x29,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x2B,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x94,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x98,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x9A,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x9C,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x9D,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xA0,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0xA5,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0xA7,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xA9,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xAA,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xB0,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x95,0x00,0x00,0x00,
  0x27,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x96,0x00,0x00,0x00,
  0x28,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x97,0x00,0x00,0x00,
  0x95,0x00,0x00,0x00,
  0x96,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x99,0x00,0x00,0x00,
  0x25,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x98,0x00,0x00,0x00,
  0x99,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x9B,0x00,0x00,0x00,
  0x26,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x9A,0x00,0x00,0x00,
  0x9B,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x9C,0x00,0x00,0x00,
  0x97,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x9E,0x00,0x00,0x00,
  0x29,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x9D,0x00,0x00,0x00,
  0x9E,0x00,0x00,0x00,
  0x39,0x00,0x08,0x00,
  0x06,0x00,0x00,0x00,
  0x9F,0x00,0x00,0x00,
  0x22,0x00,0x00,0x00,
  0x98,0x00,0x00,0x00,
  0x9A,0x00,0x00,0x00,
  0x9C,0x00,0x00,0x00,
  0x9D,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x94,0x00,0x00,0x00,
  0x9F,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xA1,0x00,0x00,0x00,
  0x27,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xA2,0x00,0x00,0x00,
  0x28,0x00,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xA3,0x00,0x00,0x00,
  0xA1,0x00,0x00,0x00,
  0xA2,0x00,0x00,0x00,
  0x0C,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0xA4,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x28,0x00,0x00,0x00,
  0xA3,0x00,0x00,0x00,
  0x3E,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xA6,0x00,0x00,0x00,
  0x25,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xA5,0x00,0x00,0x00,
  0xA6,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xA8,0x00,0x00,0x00,
  0x26,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xA7,0x00,0x00,0x00,
  0xA8,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xA9,0x00,0x00,0x00,
  0xA4,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xAB,0x00,0x00,0x00,
  0x29,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xAA,0x00,0x00,0x00,
  0xAB,0x00,0x00,0x00,
  0x39,0x00,0x08,0x00,
  0x06,0x00,0x00,0x00,
  0xAC,0x00,0x00,0x00,
  0x22,0x00,0x00,0x00,
  0xA5,0x00,0x00,0x00,
  0xA7,0x00,0x00,0x00,
  0xA9,0x00,0x00,0x00,
  0xAA,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xA0,0x00,0x00,0x00,
  0xAC,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xAD,0x00,0x00,0x00,
  0x94,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xAE,0x00,0x00,0x00,
  0xA0,0x00,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xAF,0x00,0x00,0x00,
  0xAD,0x00,0x00,0x00,
  0xAE,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xB0,0x00,0x00,0x00,
  0xAF,0x00,0x00,0x00,
  0x39,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xB1,0x00,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0xB0,0x00,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0xB1,0x00,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x33,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x2C,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x2D,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x2E,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x2F,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x30,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x31,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x32,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x34,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0xB4,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xC3,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xC9,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0xD2,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0xD4,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xD5,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xD7,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xB5,0x00,0x00,0x00,
  0x31,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xB4,0x00,0x00,0x00,
  0xB5,0x00,0x00,0x00,
  0x39,0x00,0x05,0x00,
  0x18,0x00,0x00,0x00,
  0xB6,0x00,0x00,0x00,
  0x1B,0x00,0x00,0x00,
  0xB4,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xB7,0x00,0x00,0x00,
  0x30,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xB8,0x00,0x00,0x00,
  0x2E,0x00,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0xB9,0x00,0x00,0x00,
  0xB7,0x00,0x00,0x00,
  0xB8,0x00,0x00,0x00,
  0x91,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0xBA,0x00,0x00,0x00,
  0xB6,0x00,0x00,0x00,
  0xB9,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xBB,0x00,0x00,0x00,
  0x30,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0xBC,0x00,0x00,0x00,
  0xBA,0x00,0x00,0x00,
  0xBB,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x2E,0x00,0x00,0x00,
  0xBC,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xBD,0x00,0x00,0x00,
  0x2E,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xBE,0x00,0x00,0x00,
  0x32,0x00,0x00,0x00,
  0x50,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0xBF,0x00,0x00,0x00,
  0xBE,0x00,0x00,0x00,
  0xBE,0x00,0x00,0x00,
  0x8D,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0xC0,0x00,0x00,0x00,
  0xBD,0x00,0x00,0x00,
  0xBF,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xC1,0x00,0x00,0x00,
  0x2D,0x00,0x00,0x00,
  0x88,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0xC2,0x00,0x00,0x00,
  0xC0,0x00,0x00,0x00,
  0xC1,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x2E,0x00,0x00,0x00,
  0xC2,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xC4,0x00,0x00,0x00,
  0x32,0x00,0x00,0x00,
  0x41,0x00,0x05,0x00,
  0x07,0x00,0x00,0x00,
  0xC5,0x00,0x00,0x00,
  0x2D,0x00,0x00,0x00,
  0x59,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xC6,0x00,0x00,0x00,
  0xC5,0x00,0x00,0x00,
  0x88,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xC7,0x00,0x00,0x00,
  0xC4,0x00,0x00,0x00,
  0xC6,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xC8,0x00,0x00,0x00,
  0xC7,0x00,0x00,0x00,
  0x83,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xC3,0x00,0x00,0x00,
  0xC8,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xCB,0x00,0x00,0x00,
  0xC3,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xCC,0x00,0x00,0x00,
  0xCA,0x00,0x00,0x00,
  0xCB,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xC9,0x00,0x00,0x00,
  0xCC,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xCD,0x00,0x00,0x00,
  0xC3,0x00,0x00,0x00,
  0x50,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0xCE,0x00,0x00,0x00,
  0xCD,0x00,0x00,0x00,
  0xCD,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xCF,0x00,0x00,0x00,
  0xC9,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xD1,0x00,0x00,0x00,
  0xCF,0x00,0x00,0x00,
  0xD0,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xD3,0x00,0x00,0x00,
  0x2E,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xD2,0x00,0x00,0x00,
  0xD3,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xD4,0x00,0x00,0x00,
  0xCE,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xD6,0x00,0x00,0x00,
  0xC9,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xD5,0x00,0x00,0x00,
  0xD6,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xD7,0x00,0x00,0x00,
  0xD1,0x00,0x00,0x00,
  0x39,0x00,0x08,0x00,
  0x06,0x00,0x00,0x00,
  0xD8,0x00,0x00,0x00,
  0x22,0x00,0x00,0x00,
  0xD2,0x00,0x00,0x00,
  0xD4,0x00,0x00,0x00,
  0xD5,0x00,0x00,0x00,
  0xD7,0x00,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0xD8,0x00,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x38,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x35,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x36,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x07,0x00,0x00,0x00,
  0x37,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x39,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xDB,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0xDC,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xDF,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xE1,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xE3,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xE4,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xF0,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xF1,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xF2,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0xFF,0x00,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x00,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x01,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x0D,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x0E,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x0F,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x13,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0xDD,0x00,0x00,0x00,
  0x36,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xDC,0x00,0x00,0x00,
  0xDD,0x00,0x00,0x00,
  0x39,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xDE,0x00,0x00,0x00,
  0x10,0x00,0x00,0x00,
  0xDC,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xDB,0x00,0x00,0x00,
  0xDE,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xE2,0x00,0x00,0x00,
  0xDB,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xE1,0x00,0x00,0x00,
  0xE2,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xE3,0x00,0x00,0x00,
  0x3E,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xE4,0x00,0x00,0x00,
  0xE0,0x00,0x00,0x00,
  0x39,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0xE5,0x00,0x00,0x00,
  0x16,0x00,0x00,0x00,
  0xE1,0x00,0x00,0x00,
  0xE3,0x00,0x00,0x00,
  0xE4,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xDF,0x00,0x00,0x00,
  0xE5,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xE6,0x00,0x00,0x00,
  0xDB,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xE8,0x00,0x00,0x00,
  0x37,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xEA,0x00,0x00,0x00,
  0xE8,0x00,0x00,0x00,
  0xE9,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xEB,0x00,0x00,0x00,
  0xE7,0x00,0x00,0x00,
  0xEA,0x00,0x00,0x00,
  0x0C,0x00,0x06,0x00,
  0x06,0x00,0x00,0x00,
  0xEC,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0xEB,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xED,0x00,0x00,0x00,
  0xE6,0x00,0x00,0x00,
  0xEC,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xF0,0x00,0x00,0x00,
  0xED,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xF1,0x00,0x00,0x00,
  0xEE,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xF2,0x00,0x00,0x00,
  0xEF,0x00,0x00,0x00,
  0x39,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0xF3,0x00,0x00,0x00,
  0x16,0x00,0x00,0x00,
  0xF0,0x00,0x00,0x00,
  0xF1,0x00,0x00,0x00,
  0xF2,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xF4,0x00,0x00,0x00,
  0xDF,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xF5,0x00,0x00,0x00,
  0xF4,0x00,0x00,0x00,
  0xF3,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xDF,0x00,0x00,0x00,
  0xF5,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xF6,0x00,0x00,0x00,
  0xDB,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0xF7,0x00,0x00,0x00,
  0x37,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xF9,0x00,0x00,0x00,
  0xF7,0x00,0x00,0x00,
  0xF8,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xFA,0x00,0x00,0x00,
  0xE7,0x00,0x00,0x00,
  0xF9,0x00,0x00,0x00,
  0x0C,0x00,0x06,0x00,
  0x06,0x00,0x00,0x00,
  0xFB,0x00,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0xFA,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0xFC,0x00,0x00,0x00,
  0xF6,0x00,0x00,0x00,
  0xFB,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xFF,0x00,0x00,0x00,
  0xFC,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x00,0x01,0x00,0x00,
  0xFD,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x01,0x01,0x00,0x00,
  0xFE,0x00,0x00,0x00,
  0x39,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0x02,0x01,0x00,0x00,
  0x16,0x00,0x00,0x00,
  0xFF,0x00,0x00,0x00,
  0x00,0x01,0x00,0x00,
  0x01,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x03,0x01,0x00,0x00,
  0xDF,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x04,0x01,0x00,0x00,
  0x03,0x01,0x00,0x00,
  0x02,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xDF,0x00,0x00,0x00,
  0x04,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x05,0x01,0x00,0x00,
  0xDB,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x06,0x01,0x00,0x00,
  0x37,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x08,0x01,0x00,0x00,
  0x06,0x01,0x00,0x00,
  0x07,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x09,0x01,0x00,0x00,
  0xE7,0x00,0x00,0x00,
  0x08,0x01,0x00,0x00,
  0x0C,0x00,0x06,0x00,
  0x06,0x00,0x00,0x00,
  0x0A,0x01,0x00,0x00,
  0x01,0x00,0x00,0x00,
  0x0D,0x00,0x00,0x00,
  0x09,0x01,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x0B,0x01,0x00,0x00,
  0x05,0x01,0x00,0x00,
  0x0A,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x0D,0x01,0x00,0x00,
  0x0B,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x0E,0x01,0x00,0x00,
  0x0C,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x0F,0x01,0x00,0x00,
  0xE9,0x00,0x00,0x00,
  0x39,0x00,0x07,0x00,
  0x06,0x00,0x00,0x00,
  0x10,0x01,0x00,0x00,
  0x16,0x00,0x00,0x00,
  0x0D,0x01,0x00,0x00,
  0x0E,0x01,0x00,0x00,
  0x0F,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x11,0x01,0x00,0x00,
  0xDF,0x00,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x12,0x01,0x00,0x00,
  0x11,0x01,0x00,0x00,
  0x10,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0xDF,0x00,0x00,0x00,
  0x12,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x14,0x01,0x00,0x00,
  0xDF,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x13,0x01,0x00,0x00,
  0x14,0x01,0x00,0x00,
  0x39,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x15,0x01,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0x13,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x17,0x01,0x00,0x00,
  0x15,0x01,0x00,0x00,
  0x16,0x01,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0x17,0x01,0x00,0x00,
  0x38,0x00,0x01,0x00,
  0x36,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x3B,0x00,0x00,0x00,
  0x00,0x00,0x00,0x00,
  0x0E,0x00,0x00,0x00,
  0x37,0x00,0x03,0x00,
  0x0D,0x00,0x00,0x00,
  0x3A,0x00,0x00,0x00,
  0xF8,0x00,0x02,0x00,
  0x3C,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x1A,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x1F,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x26,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x27,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x29,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x2B,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x2D,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x2F,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x31,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x34,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x35,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x37,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x39,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x3B,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x3F,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x43,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x44,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x46,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x48,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x0D,0x00,0x00,0x00,
  0x4A,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x4C,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x4E,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3B,0x00,0x04,0x00,
  0x07,0x00,0x00,0x00,
  0x5B,0x01,0x00,0x00,
  0x07,0x00,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x1B,0x01,0x00,0x00,
  0x3A,0x00,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x0C,0x00,0x00,0x00,
  0x1E,0x01,0x00,0x00,
  0x1B,0x01,0x00,0x00,
  0x1D,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x1A,0x01,0x00,0x00,
  0x1E,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x26,0x01,0x00,0x00,
  0x1D,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x28,0x01,0x00,0x00,
  0x1A,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x27,0x01,0x00,0x00,
  0x28,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x2A,0x01,0x00,0x00,
  0x21,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x29,0x01,0x00,0x00,
  0x2A,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x2C,0x01,0x00,0x00,
  0x23,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x2B,0x01,0x00,0x00,
  0x2C,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x2E,0x01,0x00,0x00,
  0x24,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x2D,0x01,0x00,0x00,
  0x2E,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x2F,0x01,0x00,0x00,
  0x25,0x01,0x00,0x00,
  0x39,0x00,0x0A,0x00,
  0x06,0x00,0x00,0x00,
  0x30,0x01,0x00,0x00,
  0x33,0x00,0x00,0x00,
  0x26,0x01,0x00,0x00,
  0x27,0x01,0x00,0x00,
  0x29,0x01,0x00,0x00,
  0x2B,0x01,0x00,0x00,
  0x2D,0x01,0x00,0x00,
  0x2F,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x1F,0x01,0x00,0x00,
  0x30,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x34,0x01,0x00,0x00,
  0x1D,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x36,0x01,0x00,0x00,
  0x1A,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x35,0x01,0x00,0x00,
  0x36,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x38,0x01,0x00,0x00,
  0x21,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x37,0x01,0x00,0x00,
  0x38,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x3A,0x01,0x00,0x00,
  0x32,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x39,0x01,0x00,0x00,
  0x3A,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x3C,0x01,0x00,0x00,
  0x33,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x3B,0x01,0x00,0x00,
  0x3C,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x3D,0x01,0x00,0x00,
  0xFD,0x00,0x00,0x00,
  0x39,0x00,0x0A,0x00,
  0x06,0x00,0x00,0x00,
  0x3E,0x01,0x00,0x00,
  0x33,0x00,0x00,0x00,
  0x34,0x01,0x00,0x00,
  0x35,0x01,0x00,0x00,
  0x37,0x01,0x00,0x00,
  0x39,0x01,0x00,0x00,
  0x3B,0x01,0x00,0x00,
  0x3D,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x31,0x01,0x00,0x00,
  0x3E,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x43,0x01,0x00,0x00,
  0x1D,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x45,0x01,0x00,0x00,
  0x1A,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x44,0x01,0x00,0x00,
  0x45,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x47,0x01,0x00,0x00,
  0x21,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x46,0x01,0x00,0x00,
  0x47,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x49,0x01,0x00,0x00,
  0x40,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x48,0x01,0x00,0x00,
  0x49,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x0C,0x00,0x00,0x00,
  0x4B,0x01,0x00,0x00,
  0x41,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x4A,0x01,0x00,0x00,
  0x4B,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x4C,0x01,0x00,0x00,
  0x42,0x01,0x00,0x00,
  0x39,0x00,0x0A,0x00,
  0x06,0x00,0x00,0x00,
  0x4D,0x01,0x00,0x00,
  0x33,0x00,0x00,0x00,
  0x43,0x01,0x00,0x00,
  0x44,0x01,0x00,0x00,
  0x46,0x01,0x00,0x00,
  0x48,0x01,0x00,0x00,
  0x4A,0x01,0x00,0x00,
  0x4C,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x3F,0x01,0x00,0x00,
  0x4D,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x4F,0x01,0x00,0x00,
  0x1F,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x50,0x01,0x00,0x00,
  0x1F,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x51,0x01,0x00,0x00,
  0x4F,0x01,0x00,0x00,
  0x50,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x52,0x01,0x00,0x00,
  0x31,0x01,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x53,0x01,0x00,0x00,
  0x51,0x01,0x00,0x00,
  0x52,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x54,0x01,0x00,0x00,
  0x3F,0x01,0x00,0x00,
  0x83,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x55,0x01,0x00,0x00,
  0x53,0x01,0x00,0x00,
  0x54,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x56,0x01,0x00,0x00,
  0x55,0x01,0x00,0x00,
  0x83,0x00,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x4E,0x01,0x00,0x00,
  0x56,0x01,0x00,0x00,
  0x3D,0x00,0x04,0x00,
  0x06,0x00,0x00,0x00,
  0x58,0x01,0x00,0x00,
  0x4E,0x01,0x00,0x00,
  0x85,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x59,0x01,0x00,0x00,
  0x1C,0x01,0x00,0x00,
  0x58,0x01,0x00,0x00,
  0x81,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x5A,0x01,0x00,0x00,
  0x57,0x01,0x00,0x00,
  0x59,0x01,0x00,0x00,
  0x3E,0x00,0x03,0x00,
  0x5B,0x01,0x00,0x00,
  0x5A,0x01,0x00,0x00,
  0x39,0x00,0x05,0x00,
  0x06,0x00,0x00,0x00,
  0x5C,0x01,0x00,0x00,
  0x0A,0x00,0x00,0x00,
  0x5B,0x01,0x00,0x00,
  0xFE,0x00,0x02,0x00,
  0x5C,0x01,0x00,0x00,
  0x38,0x00,0x01,0x00,
];
