/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.math.linear;

import org.apache.commons.math.MathRuntimeException;
import org.apache.commons.math.MaxIterationsExceededException;
import org.apache.commons.math.exception.util.LocalizedFormats;
import org.apache.commons.math.linear.ArrayRealVector;
import org.apache.commons.math.linear.DecompositionSolver;
import org.apache.commons.math.linear.EigenDecomposition;
import org.apache.commons.math.linear.InvalidMatrixException;
import org.apache.commons.math.linear.MatrixUtils;
import org.apache.commons.math.linear.RealMatrix;
import org.apache.commons.math.linear.RealVector;
import org.apache.commons.math.linear.SingularMatrixException;
import org.apache.commons.math.linear.TriDiagonalTransformer;
import org.apache.commons.math.util.FastMath;

public class EigenDecompositionImpl
implements EigenDecomposition {
    private byte maxIter = (byte)30;
    private double[] main;
    private double[] secondary;
    private TriDiagonalTransformer transformer;
    private double[] realEigenvalues;
    private double[] imagEigenvalues;
    private ArrayRealVector[] eigenvectors;
    private RealMatrix cachedV;
    private RealMatrix cachedD;
    private RealMatrix cachedVt;

    public EigenDecompositionImpl(RealMatrix matrix, double splitTolerance) throws InvalidMatrixException {
        if (!this.isSymmetric(matrix)) {
            throw new InvalidMatrixException(LocalizedFormats.ASSYMETRIC_EIGEN_NOT_SUPPORTED, new Object[0]);
        }
        this.transformToTridiagonal(matrix);
        this.findEigenVectors(this.transformer.getQ().getData());
    }

    public EigenDecompositionImpl(double[] main, double[] secondary, double splitTolerance) throws InvalidMatrixException {
        this.main = (double[])main.clone();
        this.secondary = (double[])secondary.clone();
        this.transformer = null;
        int size = main.length;
        double[][] z = new double[size][size];
        for (int i2 = 0; i2 < size; ++i2) {
            z[i2][i2] = 1.0;
        }
        this.findEigenVectors(z);
    }

    private boolean isSymmetric(RealMatrix matrix) {
        int rows = matrix.getRowDimension();
        int columns = matrix.getColumnDimension();
        double eps = (double)(10 * rows * columns) * (double)1.110223E-16f;
        for (int i2 = 0; i2 < rows; ++i2) {
            for (int j = i2 + 1; j < columns; ++j) {
                double mji;
                double mij = matrix.getEntry(i2, j);
                if (!(FastMath.abs(mij - (mji = matrix.getEntry(j, i2))) > FastMath.max(FastMath.abs(mij), FastMath.abs(mji)) * eps)) continue;
                return false;
            }
        }
        return true;
    }

    public RealMatrix getV() throws InvalidMatrixException {
        if (this.cachedV == null) {
            int m = this.eigenvectors.length;
            this.cachedV = MatrixUtils.createRealMatrix(m, m);
            for (int k = 0; k < m; ++k) {
                this.cachedV.setColumnVector(k, this.eigenvectors[k]);
            }
        }
        return this.cachedV;
    }

    public RealMatrix getD() throws InvalidMatrixException {
        if (this.cachedD == null) {
            this.cachedD = MatrixUtils.createRealDiagonalMatrix(this.realEigenvalues);
        }
        return this.cachedD;
    }

    public RealMatrix getVT() throws InvalidMatrixException {
        if (this.cachedVt == null) {
            int m = this.eigenvectors.length;
            this.cachedVt = MatrixUtils.createRealMatrix(m, m);
            for (int k = 0; k < m; ++k) {
                this.cachedVt.setRowVector(k, this.eigenvectors[k]);
            }
        }
        return this.cachedVt;
    }

    public double[] getRealEigenvalues() throws InvalidMatrixException {
        return (double[])this.realEigenvalues.clone();
    }

    public double getRealEigenvalue(int i2) throws InvalidMatrixException, ArrayIndexOutOfBoundsException {
        return this.realEigenvalues[i2];
    }

    public double[] getImagEigenvalues() throws InvalidMatrixException {
        return (double[])this.imagEigenvalues.clone();
    }

    public double getImagEigenvalue(int i2) throws InvalidMatrixException, ArrayIndexOutOfBoundsException {
        return this.imagEigenvalues[i2];
    }

    public RealVector getEigenvector(int i2) throws InvalidMatrixException, ArrayIndexOutOfBoundsException {
        return this.eigenvectors[i2].copy();
    }

    public double getDeterminant() {
        double determinant = 1.0;
        for (double lambda : this.realEigenvalues) {
            determinant *= lambda;
        }
        return determinant;
    }

    public DecompositionSolver getSolver() {
        return new Solver(this.realEigenvalues, this.imagEigenvalues, this.eigenvectors);
    }

    private void transformToTridiagonal(RealMatrix matrix) {
        this.transformer = new TriDiagonalTransformer(matrix);
        this.main = this.transformer.getMainDiagonalRef();
        this.secondary = this.transformer.getSecondaryDiagonalRef();
    }

    private void findEigenVectors(double[][] householderMatrix) {
        int i2;
        double[][] z = (double[][])householderMatrix.clone();
        int n = this.main.length;
        this.realEigenvalues = new double[n];
        this.imagEigenvalues = new double[n];
        double[] e = new double[n];
        for (int i3 = 0; i3 < n - 1; ++i3) {
            this.realEigenvalues[i3] = this.main[i3];
            e[i3] = this.secondary[i3];
        }
        this.realEigenvalues[n - 1] = this.main[n - 1];
        e[n - 1] = 0.0;
        double maxAbsoluteValue = 0.0;
        for (i2 = 0; i2 < n; ++i2) {
            if (FastMath.abs(this.realEigenvalues[i2]) > maxAbsoluteValue) {
                maxAbsoluteValue = FastMath.abs(this.realEigenvalues[i2]);
            }
            if (!(FastMath.abs(e[i2]) > maxAbsoluteValue)) continue;
            maxAbsoluteValue = FastMath.abs(e[i2]);
        }
        if (maxAbsoluteValue != 0.0) {
            for (i2 = 0; i2 < n; ++i2) {
                if (FastMath.abs(this.realEigenvalues[i2]) <= (double)1.110223E-16f * maxAbsoluteValue) {
                    this.realEigenvalues[i2] = 0.0;
                }
                if (!(FastMath.abs(e[i2]) <= (double)1.110223E-16f * maxAbsoluteValue)) continue;
                e[i2] = 0.0;
            }
        }
        for (int j = 0; j < n; ++j) {
            int m;
            int its = 0;
            do {
                int i4;
                for (m = j; m < n - 1; ++m) {
                    double delta = FastMath.abs(this.realEigenvalues[m]) + FastMath.abs(this.realEigenvalues[m + 1]);
                    if (FastMath.abs(e[m]) + delta == delta) break;
                }
                if (m == j) continue;
                if (its == this.maxIter) {
                    throw new InvalidMatrixException(new MaxIterationsExceededException(this.maxIter));
                }
                ++its;
                double q = (this.realEigenvalues[j + 1] - this.realEigenvalues[j]) / (2.0 * e[j]);
                double t = FastMath.sqrt(1.0 + q * q);
                q = q < 0.0 ? this.realEigenvalues[m] - this.realEigenvalues[j] + e[j] / (q - t) : this.realEigenvalues[m] - this.realEigenvalues[j] + e[j] / (q + t);
                double u = 0.0;
                double s = 1.0;
                double c = 1.0;
                for (i4 = m - 1; i4 >= j; --i4) {
                    double p = s * e[i4];
                    double h = c * e[i4];
                    if (FastMath.abs(p) >= FastMath.abs(q)) {
                        c = q / p;
                        t = FastMath.sqrt(c * c + 1.0);
                        e[i4 + 1] = p * t;
                        s = 1.0 / t;
                        c *= s;
                    } else {
                        s = p / q;
                        t = FastMath.sqrt(s * s + 1.0);
                        e[i4 + 1] = q * t;
                        c = 1.0 / t;
                        s *= c;
                    }
                    if (e[i4 + 1] == 0.0) {
                        int n2 = i4 + 1;
                        this.realEigenvalues[n2] = this.realEigenvalues[n2] - u;
                        e[m] = 0.0;
                        break;
                    }
                    q = this.realEigenvalues[i4 + 1] - u;
                    t = (this.realEigenvalues[i4] - q) * s + 2.0 * c * h;
                    u = s * t;
                    this.realEigenvalues[i4 + 1] = q + u;
                    q = c * t - h;
                    for (int ia = 0; ia < n; ++ia) {
                        p = z[ia][i4 + 1];
                        z[ia][i4 + 1] = s * z[ia][i4] + c * p;
                        z[ia][i4] = c * z[ia][i4] - s * p;
                    }
                }
                if (t == 0.0 && i4 >= j) continue;
                int n3 = j;
                this.realEigenvalues[n3] = this.realEigenvalues[n3] - u;
                e[j] = q;
                e[m] = 0.0;
            } while (m != j);
        }
        for (i2 = 0; i2 < n; ++i2) {
            int j;
            int k = i2;
            double p = this.realEigenvalues[i2];
            for (j = i2 + 1; j < n; ++j) {
                if (!(this.realEigenvalues[j] > p)) continue;
                k = j;
                p = this.realEigenvalues[j];
            }
            if (k == i2) continue;
            this.realEigenvalues[k] = this.realEigenvalues[i2];
            this.realEigenvalues[i2] = p;
            for (j = 0; j < n; ++j) {
                p = z[j][i2];
                z[j][i2] = z[j][k];
                z[j][k] = p;
            }
        }
        maxAbsoluteValue = 0.0;
        for (i2 = 0; i2 < n; ++i2) {
            if (!(FastMath.abs(this.realEigenvalues[i2]) > maxAbsoluteValue)) continue;
            maxAbsoluteValue = FastMath.abs(this.realEigenvalues[i2]);
        }
        if (maxAbsoluteValue != 0.0) {
            for (i2 = 0; i2 < n; ++i2) {
                if (!(FastMath.abs(this.realEigenvalues[i2]) < (double)1.110223E-16f * maxAbsoluteValue)) continue;
                this.realEigenvalues[i2] = 0.0;
            }
        }
        this.eigenvectors = new ArrayRealVector[n];
        double[] tmp = new double[n];
        for (int i5 = 0; i5 < n; ++i5) {
            for (int j = 0; j < n; ++j) {
                tmp[j] = z[j][i5];
            }
            this.eigenvectors[i5] = new ArrayRealVector(tmp);
        }
    }

    private static class Solver
    implements DecompositionSolver {
        private double[] realEigenvalues;
        private double[] imagEigenvalues;
        private final ArrayRealVector[] eigenvectors;

        private Solver(double[] realEigenvalues, double[] imagEigenvalues, ArrayRealVector[] eigenvectors) {
            this.realEigenvalues = realEigenvalues;
            this.imagEigenvalues = imagEigenvalues;
            this.eigenvectors = eigenvectors;
        }

        public double[] solve(double[] b) throws IllegalArgumentException, InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            if (b.length != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.VECTOR_LENGTH_MISMATCH, b.length, m);
            }
            double[] bp = new double[m];
            for (int i2 = 0; i2 < m; ++i2) {
                ArrayRealVector v = this.eigenvectors[i2];
                double[] vData = v.getDataRef();
                double s = v.dotProduct(b) / this.realEigenvalues[i2];
                for (int j = 0; j < m; ++j) {
                    int n = j;
                    bp[n] = bp[n] + s * vData[j];
                }
            }
            return bp;
        }

        public RealVector solve(RealVector b) throws IllegalArgumentException, InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            if (b.getDimension() != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.VECTOR_LENGTH_MISMATCH, b.getDimension(), m);
            }
            double[] bp = new double[m];
            for (int i2 = 0; i2 < m; ++i2) {
                ArrayRealVector v = this.eigenvectors[i2];
                double[] vData = v.getDataRef();
                double s = v.dotProduct(b) / this.realEigenvalues[i2];
                for (int j = 0; j < m; ++j) {
                    int n = j;
                    bp[n] = bp[n] + s * vData[j];
                }
            }
            return new ArrayRealVector(bp, false);
        }

        public RealMatrix solve(RealMatrix b) throws IllegalArgumentException, InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            if (b.getRowDimension() != m) {
                throw MathRuntimeException.createIllegalArgumentException(LocalizedFormats.DIMENSIONS_MISMATCH_2x2, b.getRowDimension(), b.getColumnDimension(), m, "n");
            }
            int nColB = b.getColumnDimension();
            double[][] bp = new double[m][nColB];
            for (int k = 0; k < nColB; ++k) {
                for (int i2 = 0; i2 < m; ++i2) {
                    int j;
                    ArrayRealVector v = this.eigenvectors[i2];
                    double[] vData = v.getDataRef();
                    double s = 0.0;
                    for (j = 0; j < m; ++j) {
                        s += v.getEntry(j) * b.getEntry(j, k);
                    }
                    s /= this.realEigenvalues[i2];
                    for (j = 0; j < m; ++j) {
                        double[] dArray = bp[j];
                        int n = k;
                        dArray[n] = dArray[n] + s * vData[j];
                    }
                }
            }
            return MatrixUtils.createRealMatrix(bp);
        }

        public boolean isNonSingular() {
            for (int i2 = 0; i2 < this.realEigenvalues.length; ++i2) {
                if (this.realEigenvalues[i2] != 0.0 || this.imagEigenvalues[i2] != 0.0) continue;
                return false;
            }
            return true;
        }

        public RealMatrix getInverse() throws InvalidMatrixException {
            if (!this.isNonSingular()) {
                throw new SingularMatrixException();
            }
            int m = this.realEigenvalues.length;
            double[][] invData = new double[m][m];
            for (int i2 = 0; i2 < m; ++i2) {
                double[] invI = invData[i2];
                for (int j = 0; j < m; ++j) {
                    double invIJ = 0.0;
                    for (int k = 0; k < m; ++k) {
                        double[] vK = this.eigenvectors[k].getDataRef();
                        invIJ += vK[i2] * vK[j] / this.realEigenvalues[k];
                    }
                    invI[j] = invIJ;
                }
            }
            return MatrixUtils.createRealMatrix(invData);
        }
    }
}

